package com.myapp.videotools.misc;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.*;
import java.util.stream.Collectors;


@SuppressWarnings("unused")
public final class Util {

    private static final Logger LOG = LoggerFactory.getLogger(Util.class);
    private static final Map<Float, String> NAMED_RESOLUTIONS;

    static {
        Map<Float, String> namedResolutions = new LinkedHashMap<>();
        namedResolutions.put( 240.0f,  "240p");
        namedResolutions.put( 360.0f,  "360p");
        namedResolutions.put( 480.0f,  "480p");
        namedResolutions.put( 720.0f,  "720p");
        namedResolutions.put(1080.0f, "1080p");
        namedResolutions.put(2160.0f,    "4K");
        namedResolutions.put(4320.0f,    "8K");
        namedResolutions.put(8640.0f,   "16K");
        NAMED_RESOLUTIONS = Collections.unmodifiableMap(namedResolutions);
    }

    private Util() {
        // utility class
    }

    public static String getOutput(Process p) throws IOException {
        String delimiter = System.getProperty("line.separator");

        String out = IOUtils.readLines(p.getInputStream(), Charset.defaultCharset()).stream()
                .peek(s -> LOG.trace("          [avconv output] : " + s))
                .collect(Collectors.joining(delimiter))
                .trim();

        String err = IOUtils.readLines(p.getErrorStream(), Charset.defaultCharset()).stream()
                .peek(s -> LOG.trace("          [avconv stderr] : " + s))
                .collect(Collectors.joining(delimiter))
                .trim();
        return "----- STDOUT -----\n" + out +"\n----- STDERR -----\n" + err +"\n------------------" ;
    }

    public static String formatOffset(long offsetMillis) {
        // XXX workaround, must be greater than 0
        final long milliseconds = Math.max(offsetMillis, 1L);

        final long ms = milliseconds % 1000;

        final long seconds = milliseconds / 1000;
        final long ss = seconds % 60;

        final long minutes = milliseconds / (1000 * 60);
        final long mm = minutes % 60;

        long hours = (milliseconds / (1000 * 60 * 60));


        StringBuilder bui = new StringBuilder(16);

        if (hours < 10)
            bui.append("0");
        bui.append(hours);
        bui.append(":");

        if (mm < 10)
            bui.append("0");
        bui.append(mm);
        bui.append(":");

        if (ss < 10)
            bui.append("0");
        bui.append(ss);

        if (ms > 0) {
            bui.append(".");
            bui.append(ms);
        }

        return bui.toString();
    }

    public static final class DirsFirstAlphabeticFileComparator implements Comparator<File> {
        @Override
        public int compare(File o1, File o2) {
            if (o1.equals(o2))
                return 0;
            
            boolean o1IsDir = o1.isDirectory(), o2IsDir = o2.isDirectory();
            
            if (o1IsDir && ! o2IsDir) 
                return -1;
            if (o2IsDir && ! o1IsDir) 
                return 1;
                
            return o1.getPath().compareToIgnoreCase(o2.getPath());
        }
    }

    public static boolean isSameFile(File a, File b) {
        try {
            return Files.isSameFile(Paths.get(a.getPath()), Paths.get(b.getPath()));
        } catch (IOException e) {
            throw new RuntimeException("could not compare files " + a + " and " + b, e);
        }
    }
    

    public static String squeezeFileName(String fileName) {
        int maxUnsqueezedLength = 50;
        return squeezeFileName(maxUnsqueezedLength, fileName);
    }

    public static String squeezeFileName(int maxUnsqueezedLength, String fileName) {
        if (fileName.length() < maxUnsqueezedLength) {
            return fileName;
        }
        String suffix = "...";

        if (fileName.length() <= suffix.length() + 3) {
            return fileName.substring(0, maxUnsqueezedLength);
        }

        int lastDotIndex = fileName.lastIndexOf(".");
        if (lastDotIndex >= 0) {
            String beyondDot = fileName.substring(lastDotIndex);
            if (beyondDot.length() <= 5) { // ".mpeg"
                suffix += beyondDot;
            }
        }
        int length = Math.max(maxUnsqueezedLength - suffix.length(), 0);
        return fileName.substring(0, length) + suffix;
    }

    public static String getOsInfoString() {
        String osName = System.getProperty("os.name");
        String osVersion = System.getProperty("os.version");
        return "Operating System: '"+osName+"', Version: '"+osVersion+"'";
    }
    


    public static String printArgs(String[] a) {
        return Arrays.stream(a).map(c -> "\"" + c + "\"").collect(Collectors.joining(" "));
    }

    public static String printArgs(List<String> l) {
        return l.stream().map(c -> "\"" + c + "\"").collect(Collectors.joining(" "));
    }

    public static String getDimensionString(int w, int h) {
        if (w < 0 || h < 0) {
            return "";
        }
        return NAMED_RESOLUTIONS.entrySet().stream()
                .filter(e -> h >= e.getKey() * 0.9f && h < e.getKey() * 1.1f && w >= e.getKey())
                .map(Map.Entry::getValue)
                .findFirst()
                .orElseGet(() -> w + "x" + h);
    }
}
