package com.myapp.videotools.misc;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.LineIterator;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.myapp.videotools.commandline.CheckForDownloadableMoviesCommand.FORMATTER;

/**
 * Data holder representing a tv-show or a movie.
 *
 * @author andre
 */
@SuppressWarnings("WeakerAccess")
public class Broadcast {

    public static final String CDATA_BEGIN = "=<![CDATA[";
    public static final String CDATA_END = "]]>";
    private static final Pattern SERIALIZATION_REGEX = Pattern.compile(
            "\\b(\\S+)" + Pattern.quote(CDATA_BEGIN) + "(.*?)" + Pattern.quote(CDATA_END) + ";");

    private String title;
    private String description;
    private String duration;
    private int durationSeconds;
    private ZonedDateTime airDate;
    private String streamUrl;
    private String imageUrl;
    private String tvChannel;
    private String availability;


    @Override
    public String toString() {
        return "Broadcast{" +
                  "title=\"" + title + '\"' +
                ", duration='" + duration + '\"' +
                ", tvChannel='" + tvChannel + '\'' +
                ", availability='" + availability + '\'' +
                ", imageUrl='" + imageUrl + '\'' +
                ", description='" + description + '\'' +
                ", streamUrl='" + streamUrl + '\'' +
                ", airDate=" + airDate.format(FORMATTER) +
                '}';
    }

    public static List<Broadcast> fromFile(File broadcastInput) throws IOException {
        List<Broadcast> broadcasts = new ArrayList<>();
        LineIterator liter = FileUtils.lineIterator(broadcastInput);
        while (liter.hasNext()) {
            String line = liter.nextLine();
            broadcasts.add(fromSerializationString(line));
        }
        return broadcasts;
    }

    public static void toFile(File broadcastOutput, List<Broadcast> broadcasts) throws IOException {
        try (PrintWriter pw = new PrintWriter(new FileWriter(broadcastOutput))) {
            broadcasts.forEach(bc -> pw.println(bc.toSerializationString()));
        }
    }

    private String toSerializationString() {
        //noinspection StringBufferReplaceableByString
        StringBuilder sb = new StringBuilder();
        sb.append("airDate"        ).append("=").append(CDATA_BEGIN).append(FORMATTER.format(airDate)).append(CDATA_END).append(";");
        sb.append("title"          ).append("=").append(CDATA_BEGIN).append(title          ).append(CDATA_END).append(";");
        sb.append("description"    ).append("=").append(CDATA_BEGIN).append(description    ).append(CDATA_END).append(";");
        sb.append("duration"       ).append("=").append(CDATA_BEGIN).append(duration       ).append(CDATA_END).append(";");
        sb.append("durationSeconds").append("=").append(CDATA_BEGIN).append(durationSeconds).append(CDATA_END).append(";");
        sb.append("streamUrl"      ).append("=").append(CDATA_BEGIN).append(streamUrl      ).append(CDATA_END).append(";");
        sb.append("imageUrl"       ).append("=").append(CDATA_BEGIN).append(imageUrl       ).append(CDATA_END).append(";");
        sb.append("tvChannel"      ).append("=").append(CDATA_BEGIN).append(tvChannel      ).append(CDATA_END).append(";");
        sb.append("availability"   ).append("=").append(CDATA_BEGIN).append(availability   ).append(CDATA_END).append(";");
        return sb.toString();
    }

    private static Broadcast fromSerializationString(String serialized) {
        Broadcast bc = new Broadcast();
        Matcher m = SERIALIZATION_REGEX.matcher(serialized);
        while (m.find()) {
            String property = m.group(1);
            String value = m.group(2);
            if (value.equals("null")) {
                value = "";
            }
            switch (property) {
                case "title"          : bc.title           = value; break;
                case "description"    : bc.description     = value; break;
                case "duration"       : bc.duration        = value; break;
                case "durationSeconds": bc.durationSeconds = Integer.parseInt(value); break;
                case "streamUrl"      : bc.streamUrl       = value; break;
                case "imageUrl"       : bc.imageUrl        = value; break;
                case "tvChannel"      : bc.tvChannel       = value; break;
                case "availability"   : bc.availability    = value; break;
                case "airDate"        : bc.airDate         =
                        value.isEmpty() ? null : ZonedDateTime.parse(value, FORMATTER); break;
                default:
                    throw new RuntimeException("unknown property: '" + property + "' " +
                            "in serialization string: '" + serialized + "'");
            }
        }
        return bc;
    }

    public String getTitle() {
        return title;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public String getDuration() {
        return duration;
    }

    public void setDuration(String duration) {
        this.duration = duration;
    }

    public int getDurationSeconds() {
        return durationSeconds;
    }

    public void setDurationSeconds(int durationSeconds) {
        this.durationSeconds = durationSeconds;
    }

    @SuppressWarnings("unused")
    public ZonedDateTime getAirDate() {
        return airDate;
    }

    public void setAirDate(ZonedDateTime airDate) {
        this.airDate = airDate;
    }

    @SuppressWarnings("unused")
    public String getStreamUrl() {
        return streamUrl;
    }

    public void setStreamUrl(String streamUrl) {
        this.streamUrl = streamUrl;
    }

    @SuppressWarnings("unused")
    public String getImageUrl() {
        return imageUrl;
    }

    public void setImageUrl(String imageUrl) {
        this.imageUrl = imageUrl;
    }

    public String getTvChannel() {
        return tvChannel;
    }

    public void setTvChannel(String tvChannel) {
        this.tvChannel = tvChannel;
    }

    public String getAvailability() {
        return availability;
    }

    public void setAvailability(String availability) {
        this.availability = availability;
    }
}
