package com.myapp.videotools.misc;

import com.myapp.util.format.TimeFormatUtil;
import org.assertj.core.util.VisibleForTesting;

import java.text.DecimalFormat;
import java.text.Format;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

@SuppressWarnings({"WeakerAccess"})
public class AppStatistics {

    private static final Format DECIMAL_FORMAT= new DecimalFormat("##0.0");
    private static final String NL = System.getProperty("line.separator");
    
    private long applicationStart = -1L;
    private long applicationExit = -1L;
    private AtomicInteger thumbnailsCreated = new AtomicInteger(0);
    private AtomicInteger bigPicturesMerged = new AtomicInteger(0);
    private AtomicInteger filesSkippedBecauseExistingTarget = new AtomicInteger(0);
    private AtomicInteger filesSkippedBecauseFiltered = new AtomicInteger(0);
    private AtomicInteger filesParsed = new AtomicInteger(0);
    
    private AtomicInteger thumbnailFails = new AtomicInteger(0);
    private AtomicInteger mergeFails = new AtomicInteger(0);
    private AtomicInteger parseFails = new AtomicInteger(0);

    private AtomicLong timeSpentWithThumbnailing = new AtomicLong(0);
    private AtomicLong timeSpentWithImageMerging = new AtomicLong(0);
    private AtomicLong timeSpentWithParsingMetadata = new AtomicLong(0);

    
    private static AppStatistics instance = null;
    
    public static AppStatistics getInstance() {
        if (instance == null)
            synchronized (AppStatistics.class) {
                if (instance == null)
                    instance = new AppStatistics();
            }
        return instance;
    }
    
    
    private AppStatistics() {}

    @VisibleForTesting
    public void reset() {
        applicationStart = -1L;
        applicationExit = -1L;
        thumbnailsCreated = new AtomicInteger(0);
        bigPicturesMerged = new AtomicInteger(0);
        filesSkippedBecauseExistingTarget = new AtomicInteger(0);
        filesSkippedBecauseFiltered = new AtomicInteger(0);
        filesParsed = new AtomicInteger(0);
        thumbnailFails = new AtomicInteger(0);
        mergeFails = new AtomicInteger(0);
        parseFails = new AtomicInteger(0);
        timeSpentWithThumbnailing = new AtomicLong(0);
        timeSpentWithImageMerging = new AtomicLong(0);
        timeSpentWithParsingMetadata = new AtomicLong(0);
    }
    
    
    // total times
    
    
    public void setApplicationStart() {
        if (applicationStart != -1L)
            throw new IllegalStateException("application already started!");
        
        applicationStart = System.currentTimeMillis();
    }
    public void setApplicationExit() {
        if (applicationExit != -1L)
            throw new IllegalStateException("application already exited!");
        
        applicationExit = System.currentTimeMillis();
    }
    public long getApplicationStart() {
        return applicationStart;
    }
    public long getApplicationExit() {
        return applicationExit;
    }
    public long getTotalTimeNeeded() {
        if (applicationStart == -1 || applicationExit == -1)
            throw new IllegalStateException(applicationStart+" , "+applicationExit);
        
        return applicationExit - applicationStart;
    }
    
    
    
    

    // thumbnailing
    
    public final void addTimeSpentWithThumbnailing(long delta) {
        timeSpentWithThumbnailing.addAndGet(delta);
    }
    public final long getTimepentWithThumbnailing() {
        return timeSpentWithThumbnailing.get();
    }
    public final void addThumbnailsCreated(int delta) {
        thumbnailsCreated.addAndGet(delta);
    }
    public final int getThumbnailsCreated() {
        return thumbnailsCreated.get();
    }
    public final void incrementThumbnailsCreated() {
        thumbnailsCreated.incrementAndGet();
    }
    public final void incrementThumbnailFails() {
        thumbnailFails.incrementAndGet();
    }
    public final int getThumbnailFails() {
        return thumbnailFails.get();
    }

    
    

    // merging images
    
    
    public final void addTimeSpentWithImageMerging(long delta) {
        timeSpentWithImageMerging.addAndGet(delta);
    }
    public final long getTimeSpentWithImageMerging() {
        return timeSpentWithImageMerging.get();
    }
    public final int getBigPicturesMerged() {
        return bigPicturesMerged.get();
    }
    public final void incrementBigPicturesMerged() {
        bigPicturesMerged.incrementAndGet();
    }
    public final void incrementMergeFails() {
        mergeFails.incrementAndGet();
    }
    public final int getMergeFails() {
        return mergeFails.get();
    }
    
    
    
    
    // parsing
    
    public final void addTimeSpentWithParsingMetadata(long delta) {
        timeSpentWithParsingMetadata.addAndGet(delta);
    }
    public final long getTimeSpentWithParsingMetadata() {
        return timeSpentWithParsingMetadata.get();
    }
    public final void incrementFilesParsed() {
        filesParsed.incrementAndGet();
    }
    public final long getFilesParsed() {
        return filesParsed.get();
    }
    public final void incrementParseFails() {
        parseFails.incrementAndGet();
    }
    public final int getParseFails() {
        return parseFails.get();
    }

    
    
    // skipping
    
    
    public final int getSkippedBecauseFiltered() {
        return filesSkippedBecauseFiltered.get();
    }
    public final void incrementSkippedBecauseFiltered() {
        filesSkippedBecauseFiltered.incrementAndGet();
    }
    public final int getSkippedBecauseExistingTarget() {
        return filesSkippedBecauseExistingTarget.get();
    }
    public final void incrementSkippedBecauseExistingTarget() {
        filesSkippedBecauseExistingTarget.incrementAndGet();
    }
    
    
    

    
    @Override
    public String toString() {
        Long totalMillis        = getTotalTimeNeeded();
        
        StringBuilder b = new StringBuilder();
        b.append("ApplicationStatistics:").append(NL).append(NL);
        b.append("  creation statistics:").append(NL);

        Long parsedMillis = getTimeSpentWithParsingMetadata();
        Long mergedMillis = getTimeSpentWithImageMerging();
        Long thumbedMillis = getTimepentWithThumbnailing();

        appendCountWithTimeForEachFile(b, "video-metadata parsed", getFilesParsed(),       parsedMillis,  getParseFails());
        appendCountWithTimeForEachFile(b, "big-pictures merged",   getBigPicturesMerged(), mergedMillis,  getMergeFails());
        appendCountWithTimeForEachFile(b, "thumbnails created",    getThumbnailsCreated(), thumbedMillis, getThumbnailFails());
        
        b.append("    skipped because target file was existing       : ").append(getSkippedBecauseExistingTarget()).append(NL);
        b.append("    skipped because source file was filtered       : ").append(getSkippedBecauseFiltered()).append(NL).append(NL);
        b.append("  time statistics:").append(NL);
        
        appendTimeSpentWithPercent("with parsing metadata", totalMillis, parsedMillis, b);
        appendTimeSpentWithPercent("with thumbnailing videos", totalMillis, thumbedMillis, b);
        appendTimeSpentWithPercent("with image merging", totalMillis, mergedMillis, b);

        Long remainingMillis    = totalMillis - (mergedMillis + parsedMillis + thumbedMillis);
        appendTimeSpentWithPercent("with other activities", totalMillis, remainingMillis, b);
        
        b.append("    --------------------------------------------------------------------------").append(NL);
        b.append("    total time needed                              : ").append(TimeFormatUtil.getTimeLabel(totalMillis)).append(NL);
        b.append("    applicationExit                                : ").append(TimeFormatUtil.getDateLabel(getApplicationExit())).append(NL);
        b.append("    applicationStart                               : ").append(TimeFormatUtil.getDateLabel(applicationStart)).append(NL); 
        
        return b.toString();
    }
    
    private static void appendCountWithTimeForEachFile(StringBuilder b,
                                                       String description,
                                                       Number count,
                                                       Long millisNeeded,
                                                       int fails) {
        String timeLabel;
        
        if (count.intValue() != 0) {
            double millisDbl = millisNeeded.doubleValue();
            double countDbl = count.doubleValue();
            long millisPerItem = Double.valueOf(millisDbl / countDbl).longValue();
            timeLabel = TimeFormatUtil.getTimeLabel(millisPerItem);
        
        } else {
            timeLabel = "N/A";
        }
        
        b.append("    ");
        b.append(description);

        appendSpaces(46 - description.length(), b);

        b.append(" : ");
        b.append(count);

        appendSpaces(5 - count.toString().length(), b);

        b.append("  (");
        b.append(timeLabel);
        appendSpaces(10 - timeLabel.length(), b);
        b.append(" /file)");
        
        if (fails != 0) {
            b.append("      ");
            b.append(fails);
            b.append(" FAILS !");
        }
        
        b.append(NL);
    }
    
    private static void appendTimeSpentWithPercent(String action, Long totalTime, Long millis, StringBuilder b) {
        double percent = millis.doubleValue() / totalTime.doubleValue() * 100d;
        String percentString = DECIMAL_FORMAT.format(percent);
        String timeLabel = TimeFormatUtil.getTimeLabel(millis);
        
        b.append("    time spent ");
        b.append(action);

        int spaces = 35 - action.length();
        appendSpaces(spaces, b);

        b.append(" : ");
        b.append(timeLabel);

        appendSpaces(15 - timeLabel.length(), b);
        appendSpaces(8 - percentString.length(), b);

        b.append(percentString);
        b.append(" %");
        b.append(NL);
    }

    private static void appendSpaces(int spaces, StringBuilder b) {
        for (int i = 0; i < spaces; i++) {
            b.append(' ');
        }
    }

}
