package com.myapp.videotools.impl;

import com.myapp.util.format.FileFormatUtil;
import com.myapp.util.format.TimeFormatUtil;
import com.myapp.util.process.ProcessTimeoutKiller;
import com.myapp.videotools.IImageMerger;
import com.myapp.videotools.VideoFile;
import com.myapp.videotools.commandline.RangeSelection;
import com.myapp.videotools.misc.AppStatistics;
import com.myapp.videotools.misc.Configuration;
import com.myapp.videotools.misc.Util;
import org.assertj.core.api.Assertions;
import org.assertj.core.util.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.*;

import static com.myapp.videotools.AbstractVideoThumbnailer.calculateOffsetMillis;

/**
 * merges a bunch of thumbnail images of a video file to a nice and titled grid
 * by invoking a "montage" process. ( http://www.imagemagick.org)
 * 
 * <pre>
    andre@buenosaires ~ % montage -version
    Version: ImageMagick 6.5.7-8 2009-11-26 Q16 http://www.imagemagick.org
    Copyright: Copyright (C) 1999-2009 ImageMagick Studio LLC
    Features: OpenMP
   </pre> 
 * 
 * @author andre
 * 
 */
class ImageMagickImageMerger implements IImageMerger {

    private static final String MONTAGE_COMMAND_PROPKEY = "com.myapp.videotools.MONTAGE_CMD";

    private static final Logger log = LoggerFactory.getLogger(ImageMagickImageMerger.class);
    private static final AppStatistics statistics = AppStatistics.getInstance();

    private final String montageCommand;
    private RangeSelection rangeSelection;

    @SuppressWarnings("WeakerAccess") // invoked via reflection
    public ImageMagickImageMerger() {
        montageCommand = Configuration.getInstance().getProperty(MONTAGE_COMMAND_PROPKEY);
    }
    
    
    public void mergeImages(int rows,
                            int cols,
                            File out,
                            int tileWidth,
                            int tileHeight,
                            List<File> imageList,
                            VideoFile videoFile) throws IOException  {
        final int elementCount = imageList == null ? 0 : imageList.size();


        // specify outputfile:
        //--------------------------------------
        String outputFilePath = out.getPath();
        String nameLower = out.getName().toLowerCase();

        // TODO: think about if we really want to change the path of the output file parameter:
        if ( ! (nameLower.endsWith(".jpeg") || nameLower.endsWith(".jpg"))) {
            outputFilePath += ".jpeg";
        }

        log.info("        Merging {} pictures to:           {} ...", elementCount, out.getName());
        
        if (elementCount <= 0) {
            throw new RuntimeException("no files specified!");
        }

        List<String> args = new ArrayList<>();

        if (montageCommand.contains("%")) {
            args.addAll(Arrays.asList(montageCommand.split("[%]")));
        } else {
            args.add(montageCommand);
        }

        if (videoFile != null) {
            // specify the files with their time labels according to the videofile:
            //--------------------------------------

            for (int i = 0; i < elementCount; i++) {
                double currentTimeInSeconds = 0.001 * calculateOffsetMillis(rangeSelection, videoFile, i, elementCount);
                File f = imageList.get(i);
                args.add("-label");
                args.add(TimeFormatUtil.getTimeLabel(currentTimeInSeconds));
                args.add(f.getPath());
            }
        
        } else {
            // specify just the files:
            //--------------------------------------
            for (int i = 0; i < elementCount; i++)
                 args.add(imageList.get(i).getPath());
        }
        
        // specify layout: 
        //--------------------------------------
        args.add("-tile");
        args.add(cols+"x"); // (e.g. -tile 5 will result in rows of size 5)
        args.add("-geometry");
        args.add(tileWidth+"x"+tileHeight+"+1+1"); //  <--- NOTE, +1+1 adds 1px border to tiles

        // set background color to black:
        args.add("-background");
        args.add("black");
        args.add("-fill");
        args.add("white");

        if (videoFile != null) {
            args.add("-title");
            args.add(generateTitleString(videoFile, cols, tileWidth));
        }

        args.add(outputFilePath);

        if (log.isTraceEnabled()) log.trace("          Starting process:               {}", Util.printArgs(args));
        
        
        // start montage process:
        //--------------------------------------
        ProcessBuilder pb = new ProcessBuilder(args);
        Process p = null;
        long start = System.currentTimeMillis();

        try {
            p = pb.start();
            int i = waitForProcess(out, p);

            if (i != 0) {
                log.warn("          Process: {} returned with {}", args.get(0) , i);
                log.error("        FAIL, no pictures were merged !");
            }
            
        } catch (InterruptedException e) {
            log.error("        Could not merge thumbnails");
            log.trace("        Stacktrace:", e);
            statistics.incrementMergeFails();
            return;
            
        } finally {
            if (p != null) {
                p.destroy();
            }
            ProcessTimeoutKiller.cancelKillTimeout(p);
            log.trace("          Process cleaned up.");
            statistics.addTimeSpentWithImageMerging(System.currentTimeMillis() - start);
        }

        statistics.incrementBigPicturesMerged();
        log.info("        OK, {} pictures were merged.", elementCount);
    }

    private static String generateTitleString(VideoFile videoFile, int cols, int tileWidthPixels) {
        String videoName = videoFile.getName();
        Assertions.assertThat(videoName).as("videofile name must be set.").isNotNull();

        // specify filename + some metadata as header at top of image,
        // squeezing the title if it is not fitting into the final picture:
        //--------------------------------------
        StringBuilder titleBuilder = new StringBuilder();
        titleBuilder.append(" [");
        titleBuilder.append(FileFormatUtil.getHumanReadableFileSize(videoFile.getFile()));
        titleBuilder.append(" ");
        int w = videoFile.getVideoWidth();
        int h = videoFile.getVideoHeight();
        if (w > 0 && h > 0) {
            String dimensions = Util.getDimensionString(w, h);
            titleBuilder.append(dimensions).append(" ");
        }
        titleBuilder.append(TimeFormatUtil.getTimeLabel(videoFile.getLengthSeconds(), false));
        titleBuilder.append("]");

        // estimate how much space we have for the title.
        // for example, the string
        // "Clash of Empires Die Schlacht um Asien - 2011 (B....avi [1.56 GB 01:49:15]"
        // has 74 chars, and, measured with gimp, 789 pixels. 789.F / 74.F = 10.662162
        // lets round it up to 12:
        final float pixelsPerChar = 11F;
        int estimatedTitleChars = videoName.length() + titleBuilder.length();
        int estimatedTitlePx  = Math.round(pixelsPerChar * estimatedTitleChars);
        int mergedImageWidthPx = cols * (tileWidthPixels + 2);

        if (estimatedTitlePx > mergedImageWidthPx) {
            // see how much space we can afford to assign to the video file name:
            int beyondPx = estimatedTitlePx - mergedImageWidthPx;
            int beyondChars = Math.round(beyondPx / pixelsPerChar);
            int squeezedLengthChars = videoName.length() - beyondChars;
            videoName = Util.squeezeFileName(squeezedLengthChars, videoName);
        }

        titleBuilder.insert(0, videoName);
        return titleBuilder.toString();
    }

    @VisibleForTesting
    protected int waitForProcess(File out, Process p) throws InterruptedException {
        ProcessTimeoutKiller.registerKillTimeout(p, 10000, "merge images to: "+out.getName(), false);
        int i = p.waitFor();
        ProcessTimeoutKiller.cancelKillTimeout(p);
        return i;
    }

    @Override
    public void setRangeSelection(RangeSelection rangeSelection) {
        this.rangeSelection = rangeSelection;
    }
}
