package com.myapp.videotools.impl;

import com.myapp.videotools.AbstractPathCalculator;
import org.apache.commons.lang3.StringUtils;

import java.io.File;

/**
 * path calculator strategy that picks the target path
 * inside a target directory, keeping the relative path to the source directory.
 * <br/>
 * in other words, there are 2 file hierarchies (source root dir, target root dir)
 * and the resulting path has the same relative path to both source and target root dir.
 */
public final class FileHierarchyCopying extends AbstractPathCalculator {

     private String videoRootDir = null;
     private String targetRootDir = null;

     public FileHierarchyCopying(File videoRootDir, File targetRootDir) {
         setTargetRootDirFile(targetRootDir);
         setVideoRootDirFile(videoRootDir);
     }


     @Override
     public synchronized String getTargetFileImpl(File source) {
         String parent = source.getParentFile().getPath();
         parent = parent.replace(videoRootDir, targetRootDir);

         StringBuilder builder = new StringBuilder();
         builder.append(parent);

         if (builder.charAt(builder.length() - 1) != File.separatorChar) {
             builder.append(File.separatorChar);
         }

         String prefix = getPrefix();
         if (StringUtils.isNotEmpty(prefix)) {
             builder.append(prefix);
         }

         builder.append(source.getName());

         String fileNameSuffix = getSuffix();
         if (StringUtils.isNotEmpty(fileNameSuffix)) {
             builder.append(fileNameSuffix);
         }

         return builder.toString();
     }

     @Override
     protected void validateSourceImpl(File source) {
         if ( ! source.getPath().contains(videoRootDir)) {
             throw new IllegalStateException("sourceFile not inside source dir: " + source);
         }
     }


     public void setTargetRootDirFile(File dir) {
         if ( ! (dir.isDirectory() && dir.canWrite()) ) {
             throw new IllegalArgumentException("targetDir not a writeable directory: " + dir);
         }
         targetRootDir = dir.getPath();
     }
     public void setVideoRootDirFile(File dir) {
         if ( ! (dir.isDirectory() && dir.canRead()) ) {
             throw new IllegalArgumentException("videoDir not a readable directory: " + dir);
         }
         videoRootDir = dir.getPath();
     }
     public String getTargetRootDir() {
         return targetRootDir;
     }
     public String getVideoRootDir() {
         return videoRootDir;
     }
}
