package com.myapp.videotools.impl;

import com.myapp.videotools.AbstractVideoFileParser;
import com.myapp.videotools.misc.Util;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import static com.myapp.videotools.misc.RegexHelper.*;
import static java.util.regex.Pattern.COMMENTS;
import static java.util.regex.Pattern.compile;

/**
 * videofile parser implementation based on following ffmpeg version:
 *
 * <pre>
 * ffmpeg version 3.2.12-1~deb9u1 Copyright (c) 2000-2018 the FFmpeg developers
 *   built with gcc 6.3.0 (Debian 6.3.0-18+deb9u1) 20170516
 *   configuration: --prefix=/usr --extra-version='1~deb9u1' --toolchain=hardened --libdir=/usr/lib/x86_64-linux-gnu --incdir=/usr/include/x86_64-linux-gnu --enable-gpl --disable-stripping --enable-avresample --enable-avisynth --enable-gnutls --enable-ladspa --enable-libass --enable-libbluray --enable-libbs2b --enable-libcaca --enable-libcdio --enable-libebur128 --enable-libflite --enable-libfontconfig --enable-libfreetype --enable-libfribidi --enable-libgme --enable-libgsm --enable-libmp3lame --enable-libopenjpeg --enable-libopenmpt --enable-libopus --enable-libpulse --enable-librubberband --enable-libshine --enable-libsnappy --enable-libsoxr --enable-libspeex --enable-libssh --enable-libtheora --enable-libtwolame --enable-libvorbis --enable-libvpx --enable-libwavpack --enable-libwebp --enable-libx265 --enable-libxvid --enable-libzmq --enable-libzvbi --enable-omx --enable-openal --enable-opengl --enable-sdl2 --enable-libdc1394 --enable-libiec61883 --enable-chromaprint --enable-frei0r --enable-libopencv --enable-libx264 --enable-shared
 *   libavutil      55. 34.101 / 55. 34.101
 *   libavcodec     57. 64.101 / 57. 64.101
 *   libavformat    57. 56.101 / 57. 56.101
 *   libavdevice    57.  1.100 / 57.  1.100
 *   libavfilter     6. 65.100 /  6. 65.100
 *   libavresample   3.  1.  0 /  3.  1.  0
 *   libswscale      4.  2.100 /  4.  2.100
 *   libswresample   2.  3.100 /  2.  3.100
 *   libpostproc    54.  1.100 / 54.  1.100
 * </pre>
 */
public class FfmpegVideoFileParser extends AbstractVideoFileParser implements AvconvConstants {


    // SPECIAL RETURN VALUES FOR ERROR INDICATION:



    /**
     * decides if a line of the ffmpeg/avprobe output is interesting
     * for the parser. other lines won't be parsed.
     */
    private static final Pattern LINE_OF_INTEREST_PATTERN = compile("\\s* ( Input | Duration: | Stream ) \\s+ .* ", COMMENTS);




    /* BELOW ARE REGEXES FOR PARSING GENERAL MEDIA METADATA FROM LINES LIKE:

       Duration: 00:05:04.37, start: 0.000000, bitrate: 584 kb/s
       Duration: 00:09:59.00, start: 0.000000, bitrate: 2120 kb/s
       Duration: 00:22:13.00, start: 0.000000, bitrate: 1103 kb/s
       Duration: 00:23:11.79, start: 0.000000, bitrate: 1406 kb/s
       Duration: 00:00:00.10, start: 0.233567, bitrate: 9165 kb/s
       Duration: 00:00:00.04, start: 0.060000, bitrate: 2048 kb/s
       Duration: N/A, bitrate: N/A

    */

    /**
     * example input:
     * <pre>
     * "  Duration: 00:21:00.01,"
     * "  Duration: 00:21:37.09,"
     * "  Duration: 00:00:00.10,"
     * "  Duration: 00:00:00.00,"
     * "  Duration: N/A,"
     * </pre>
     */
    static final Pattern DURATION_PATTERN = compile("^\\s*[Dd]uration: (\\d\\d:\\d\\d:\\d\\d\\.\\d\\d), ");

    /**
     * example input:
     * <pre>
     * ", bitrate: 1280 kb/s"
     * ", bitrate: 448 kb/s"
     * ", bitrate: 1207 kb/s"
     * ", bitrate: 663 kb/s"
     *
     * </pre>
     */
    static final Pattern TOTAL_BYTES_PER_SEC_PATTERN = compile("^\\s*[Dd]uration: .*, bitrate: ([0-9]+) kb/s\\b");

    /**
     * example input:
     * <pre>
     * ", start: 0.523311,"
     * ", start: 0.777367,"
     * ", start: 0.033344,"
     * ", start: 0.062000,"
     * </pre>
     */
    static final Pattern VIDEO_START_OFFSET_PATTERN = compile("^\\s*[Dd]uration: .*,\\s*[Ss]tart\\s*:\\s*([0-9]+(?:\\.[0-9]+)), ");






    /* BELOW ARE REGEXES FOR PARSING VIDEO METADATA FROM LINES LIKE:

       Stream #0:1[0x1e0]: Video: mpeg2video (Main), yuv420p(tv, smpte170m, top first), 720x576 [SAR 16:15 DAR 4:3], 9000 kb/s, 25 tbr, 90k tbn, 50 tbc
       Stream #0:1: Video: wmv3 (Main) (WMV3 / 0x33564D57), yuv420p, 720x480, 1400 kb/s, 30 fps, 30 tbr, 1k tbn, 1k tbc
       Stream #0:0: Video: h264 (Main), yuv420p(progressive), 640x360 [SAR 1:1 DAR 16:9], 25 fps, 25 tbr, 1k tbn, 50 tbc
       Stream #0:1(eng): Video: wmv3 (Main) (WMV3 / 0x33564D57), yuv420p, 460x348, 1100 kb/s, 23 fps, 23 tbr, 1k tbn, 1k tbc
       Stream #0:0: Video: mpeg4 (Simple Profile) (divx / 0x78766964), yuv420p, 640x464 [SAR 1:1 DAR 40:29], 400 kb/s, 23.98 fps, 23.98 tbr, 23.98 tbn, 2997 tbc
       Stream #0:0: Video: vp6f, yuv420p, 720x480, 29.97 tbr, 1k tbn
       Stream #0:1(eng): Video: wmv3 (Complex) (WMV3 / 0x33564D57), yuv420p, 640x480, 1500 kb/s, 29.97 tbr, 1k tbn, 1k tbc
       Stream #0:0: Video: rawvideo (ARGB / 0x42475241), argb, 1024x600, 10 fps, 10 tbr, 10 tbn
       Stream #0:0: Video: h264 (High), yuv420p(tv, bt709/unknown/unknown, progressive), 1280x718, SAR 1:1 DAR 640:359, 23.98 fps, 23.98 tbr, 1k tbn, 47.95 tbc (default)
       Stream #0:0: Video: vp6f, yuv420p, 640x480, 30 fps, 30 tbr, 1k tbn
       Stream #0:1(bul): Video: wmv3 (Main) (WMV3 / 0x33564D57), yuv420p, 720x480, 1200 kb/s, SAR 9:10 DAR 27:20, 29.97 fps, 29.97 tbr, 1k tbn, 1k tbc
       Stream #0:0: Video: rawvideo (RGB[15] / 0xF424752), rgb555le, 1024x600, 10 fps, 10 tbr, 10 tbn
       Stream #0:0[0x1e0]: Video: mpeg1video, yuv420p(tv), 640x480 [SAR 1:1 DAR 4:3], 1136 kb/s, 29.97 fps, 29.97 tbr, 90k tbn, 29.97 tbc
       Stream #0:0[0x1e0]: Video: mpeg1video, yuv420p(tv), 640x480 [SAR 1:1 DAR 4:3], 1136 kb/s, 25 fps, 25 tbr, 90k tbn, 25 tbc
       Stream #0:0: Video: vp6f, yuv420p, 720x480, 29.97 fps, 29.97 tbr, 1k tbn
       Stream #0:0: Video: h264 (High), yuv420p(progressive), 1280x720 [SAR 1:1 DAR 16:9], 23.98 fps, 23.98 tbr, 1k tbn, 47.95 tbc (default)
       Stream #0:0: Video: mpeg4 (DX50 / 0x30355844), yuv420p, 352x288 [SAR 1:1 DAR 11:9], 525 kb/s, 25 fps, 25 tbr, 25 tbn, 30k tbc
       Stream #0:1(bul): Video: wmv3 (Main) (WMV3 / 0x33564D57), yuv420p, 720x480, 1200 kb/s, SAR 9:10 DAR 27:20, 30 fps, 30 tbr, 1k tbn, 1k tbc
       Stream #0:1(ger): Video: wmv2 (WMV2 / 0x32564D57), yuv420p, 352x288, 636 kb/s, 30 tbr, 1k tbn, 1k tbc
       Stream #0:1: Video: wmv3 (Main) (WMV3 / 0x33564D57), yuv420p, 384x288, 400 kb/s, SAR 1:1 DAR 4:3, 12.50 fps, 12.50 tbr, 1k tbn, 1k tbc

    */

    /**
     * example input:
     * <pre>
     * ", 1011 kb/s"
     * ", 926 kb/s"
     * ", 399 kb/s"
     * ", 512 kb/s"
     * ", 525 kb/s"
     * ", 1200 kb/s"
     * ", 1136 kb/s"
     * ", 400 kb/s"
     * </pre>
     */
    static final Pattern VIDEO_BYTES_PER_SECOND_PATTERN = compile("^\\s*[Ss]tream .* [Vv]ideo: .*,\\s+(\\d+)\\s*kb/s\\s*,\\s+");


    /**
     * example input:
     * <pre>
     * "Video: mpeg2video (Main),"
     * "Video: mpeg4 (XVID / 0x44495658),"
     * "Video: mpeg4 (Simple Profile) (mp4v / 0x7634706D),"
     * "Video: mpeg1video,"
     * "Video: h264 (Constrained Baseline) (avc1 / 0x31637661),"
     * "Video: h264 (Main),"
     * "Video: vp6f,"
     * "Video: h264 (Main) (avc1 / 0x31637661),"
     * "Video: msmpeg4v3 (DIV3 / 0x33564944),"
     * "Video: wmv3 (Main) (WMV3 / 0x33564D57),"
     * "Video: mpeg4 (Simple Profile) (XVID / 0x44495658),"
     * "Video: h264 (High) (avc1 / 0x31637661),"
     * "Video: mpeg4 (DX50 / 0x30355844),"
     * "Video: mjpeg,"
     * "Video: h264 (High),"
     * "Video: mpeg4 (Advanced Simple Profile) (XVID / 0x44495658),"
     * </pre>
     */
    static final Pattern VIDEO_CODEC_PATTERN = compile("^\\s*[Ss]tream .* [Vv]ideo: ([^,]+?)\\s*, ");

    /**
     * example input:
     * <pre>
     * " Video: *, rgba(pc), "
     * " Video: *, yuv410p, "
     * " Video: *, yuv420p(bt709, "
     * " Video: *, yuv420p(top first), "
     * " Video: *, yuvj420p(pc), "
     * " Video: *, yuvj422p(pc, "
     * " Video: *, pal8, "
     * " Video: *, yuv420p(tv), "
     * " Video: *, yuvj444p(pc, "
     * " Video: *, yuvj420p(pc, "
     * " Video: *, yuv420p(progressive), "
     * " Video: *, yuv420p(tv, "
     * " Video: *, yuv420p, "
     * </pre>
     */
    static final Pattern VIDEO_FORMAT_PATTERN = compile("^\\s*[Ss]tream .* [Vv]ideo: [^,]+, ([^,]+),");

    /**
     * example input:
     * <pre>
     * ", 60 fps,"
     * ", 19.99 fps,"
     * ", 14.99 fps,"
     * ", 1k fps,"
     * ", 20 fps,"
     * ", 18 fps,"
     * ", 15 fps,"
     * ", 23 fps,"
     * ", 30.30 fps,"
     * ", 12.50 fps,"
     * ", 24 fps,"
     * ", 30 fps,"
     * ", 29.97 fps,"
     * ", 25 fps,"
     * ", 23.98 fps,"
     * </pre>
     */
    static final Pattern VIDEO_FRAME_RATE_PATTERN = compile("^\\s*[Ss]tream .* [Vv]ideo: .*, (\\d+(?:\\.\\d*)?)k? fps, ");

    /**
     * example input:
     * <pre>
     30     Stream #0:0: Video: msmpeg4v3 (DIV3 / 0x33564944), yuv420p, 320x240, 421 kb/s, 29.97 fps, 29.97 tbr, 29.97 tbn, 29.97 tbc
     30     Stream #0:1(eng): Video: wmv3 (Main) (WMV3 / 0x33564D57), yuv420p, 1280x720, 4871 kb/s, 29.97 fps, 29.97 tbr, 1k tbn, 1k tbc
     31     Stream #0:1(eng): Video: h264 (High), yuv420p(progressive), 1280x720, SAR 1:1 DAR 16:9, 23.98 fps, 23.98 tbr, 1k tbn, 47.95 tbc
     32     Stream #0:0: Video: vp6f, yuv420p, 640x480, 30 tbr, 1k tbn
     34     Stream #0:0: Video: vp6f, yuv420p, 720x576, 25 tbr, 1k tbn
     35     Stream #0:0(eng): Video: h264 (High), yuv420p(progressive), 1280x720, SAR 1:1 DAR 16:9, 23.98 fps, 23.98 tbr, 1k tbn, 47.95 tbc
     35     Stream #0:0(eng): Video: h264 (High), yuv420p(tv, bt709, progressive), 1280x720 [SAR 1:1 DAR 16:9], 23.98 fps, 23.98 tbr, 1k tbn, 180k tbc (default)
     36     Stream #0:0: Video: h264 (High), yuv420p(progressive), 544x408 [SAR 1:1 DAR 4:3], 25 fps, 25 tbr, 1k tbn, 50 tbc
     36     Stream #0:0: Video: mpeg4 (DX50 / 0x30355844), yuv420p, 432x320 [SAR 1:1 DAR 27:20], 820 kb/s, 25 fps, 25 tbr, 25 tbn, 30k tbc
     36     Stream #0:0: Video: vp6f, yuv420p, 720x576, 25 fps, 25 tbr, 1k tbn
     36     Stream #0:0: Video: vp6f, yuv420p, 854x480, 29.97 tbr, 1k tbn
     38     Stream #0:0: Video: h264 (High), yuv420p(tv, bt709/unknown/unknown, progressive), 720x404 [SAR 1:1 DAR 180:101], 23.98 fps, 23.98 tbr, 1k tbn, 47.95 tbc (default)
     39     Stream #0:0: Video: h264 (High), yuv420p(tv, bt709/unknown/unknown, progressive), 720x404, SAR 1:1 DAR 180:101, 23.98 fps, 23.98 tbr, 1k tbn, 47.95 tbc (default)
     39     Stream #0:0: Video: h264 (High), yuv420p(tv, progressive), 1280x720 [SAR 1:1 DAR 16:9], 23.98 fps, 23.98 tbr, 1k tbn, 47.95 tbc (default)
     40     Stream #0:0(und): Video: h264 (Main) (avc1 / 0x31637661), yuv420p, 640x360 [SAR 1:1 DAR 16:9], 370 kb/s, 25 fps, 25 tbr, 90k tbn, 50 tbc (default)
     41     Stream #0:0: Video: mpeg4 (Advanced Simple Profile) (XVID / 0x44495658), yuv420p, 512x384 [SAR 1:1 DAR 4:3], 851 kb/s, 23.98 fps, 23.98 tbr, 23.98 tbn, 23.98 tbc
     43     Stream #0:0[0x1e0]: Video: mpeg1video, yuv420p(tv), 352x240 [SAR 200:219 DAR 880:657], 1150 kb/s, 29.97 fps, 29.97 tbr, 90k tbn, 29.97 tbc
     44     Stream #0:0: Video: h264 (High), yuv420p(tv, bt709/unknown/unknown, progressive), 1280x720, SAR 1:1 DAR 16:9, 23.98 fps, 23.98 tbr, 1k tbn, 47.95 tbc (default)
     44     Stream #0:1(ger): Video: wmv3 (Main) (WMV3 / 0x33564D57), yuv420p, 384x288, 400 kb/s, SAR 1:1 DAR 4:3, 12.50 fps, 12.50 tbr, 1k tbn, 1k tbc
     46     Stream #0:0(eng): Video: h264 (High), yuv420p(progressive), 1280x720, SAR 1:1 DAR 16:9, 23.98 fps, 23.98 tbr, 1k tbn, 47.95 tbc (default)
     46     Stream #0:0: Video: gif, bgra, 666x230, 10 tbr, 100 tbn, 100 tbc
     47     Stream #0:1[0x1e0]: Video: mpeg2video (Main), yuv420p(tv, smpte170m, top first), 720x576 [SAR 16:15 DAR 4:3], 9000 kb/s, 25 tbr, 90k tbn, 50 tbc
     50     Stream #0:1: Video: wmv3 (Main) (WMV3 / 0x33564D57), yuv420p, 720x480, 1400 kb/s, 30 fps, 30 tbr, 1k tbn, 1k tbc
     52     Stream #0:0: Video: h264 (Main), yuv420p(progressive), 640x360 [SAR 1:1 DAR 16:9], 25 fps, 25 tbr, 1k tbn, 50 tbc
     52     Stream #0:1(eng): Video: wmv3 (Main) (WMV3 / 0x33564D57), yuv420p, 460x348, 1100 kb/s, 23 fps, 23 tbr, 1k tbn, 1k tbc
     54     Stream #0:0: Video: mpeg4 (Simple Profile) (divx / 0x78766964), yuv420p, 640x464 [SAR 1:1 DAR 40:29], 400 kb/s, 23.98 fps, 23.98 tbr, 23.98 tbn, 2997 tbc
     58     Stream #0:0: Video: vp6f, yuv420p, 720x480, 29.97 tbr, 1k tbn
     58     Stream #0:1(eng): Video: wmv3 (Complex) (WMV3 / 0x33564D57), yuv420p, 640x480, 1500 kb/s, 29.97 tbr, 1k tbn, 1k tbc
     64     Stream #0:0: Video: h264 (High), yuv420p(tv, bt709/unknown/unknown, progressive), 1280x718, SAR 1:1 DAR 640:359, 23.98 fps, 23.98 tbr, 1k tbn, 47.95 tbc (default)
     64     Stream #0:0: Video: vp6f, yuv420p, 640x480, 30 fps, 30 tbr, 1k tbn
     74     Stream #0:1(bul): Video: wmv3 (Main) (WMV3 / 0x33564D57), yuv420p, 720x480, 1200 kb/s, SAR 9:10 DAR 27:20, 29.97 fps, 29.97 tbr, 1k tbn, 1k tbc
     80     Stream #0:0[0x1e0]: Video: mpeg1video, yuv420p(tv), 640x480 [SAR 1:1 DAR 4:3], 1136 kb/s, 29.97 fps, 29.97 tbr, 90k tbn, 29.97 tbc
     90     Stream #0:0[0x1e0]: Video: mpeg1video, yuv420p(tv), 640x480 [SAR 1:1 DAR 4:3], 1136 kb/s, 25 fps, 25 tbr, 90k tbn, 25 tbc
     90     Stream #0:0: Video: vp6f, yuv420p, 720x480, 29.97 fps, 29.97 tbr, 1k tbn
     96     Stream #0:0: Video: h264 (High), yuv420p(progressive), 1280x720 [SAR 1:1 DAR 16:9], 23.98 fps, 23.98 tbr, 1k tbn, 47.95 tbc (default)
    102     Stream #0:0: Video: mpeg4 (DX50 / 0x30355844), yuv420p, 352x288 [SAR 1:1 DAR 11:9], 525 kb/s, 25 fps, 25 tbr, 25 tbn, 30k tbc
    104     Stream #0:1(bul): Video: wmv3 (Main) (WMV3 / 0x33564D57), yuv420p, 720x480, 1200 kb/s, SAR 9:10 DAR 27:20, 30 fps, 30 tbr, 1k tbn, 1k tbc
    110     Stream #0:1(ger): Video: wmv2 (WMV2 / 0x32564D57), yuv420p, 352x288, 636 kb/s, 30 tbr, 1k tbn, 1k tbc
    156     Stream #0:1: Video: wmv3 (Main) (WMV3 / 0x33564D57), yuv420p, 384x288, 400 kb/s, SAR 1:1 DAR 4:3, 12.50 fps, 12.50 tbr, 1k tbn, 1k tbc
     * </pre>
     * Intersting part:
     * <pre>
     * ", 1920x1080 "
     * ", 480x640 "
     * ", 768x1114 "
     * ", 800x600 "
     * ", 352x240 "
     * ", 1024x858 "
     * ", 640x368 "
     * ", 576x768 "
     * ", 1024x768 "
     * ", 450x600 "
     * ", 720x400 "
     * ", 640x272 "
     * </pre>
     *
     */
    static final Pattern VIDEO_HEIGHT_PATTERN = compile("^\\s*[Ss]tream .* [Vv]ideo: .*, \\d+[xX](\\d+)[, ]");

    /**
     * example input:
     * <pre>
     * ", 1920x1080 "
     * ", 480x640 "
     * ", 768x1114 "
     * ", 800x600 "
     * ", 352x240 "
     * ", 1024x858 "
     * ", 640x368 "
     * ", 576x768 "
     * ", 1024x768 "
     * ", 450x600 "
     * ", 720x400 "
     * ", 640x272 "
     * </pre>
     */
    static final Pattern VIDEO_WIDTH_PATTERN = compile("^\\s*[Ss]tream .* [Vv]ideo: .*, (\\d+)[xX]\\d+[, ]");




    /* BELOW ARE REGEXES FOR PARSING AUDIO METADATA FROM LINES LIKE:

       Stream #0:1: Audio: mp3 (U[0][0][0] / 0x0055), 48000 Hz, stereo, s16p, 127 kb/s
       Stream #0:1(ger): Audio: aac (LC), 48000 Hz, stereo, fltp (default)
       Stream #0:1(und): Audio: aac (LC) (mp4a / 0x6134706D), 44100 Hz, stereo, fltp, 128 kb/s (default)
       Stream #0:1: Audio: mp3 (U[0][0][0] / 0x0055), 44100 Hz, stereo, s16p, 128 kb/s
       Stream #0:1: Audio: adpcm_swf, 44100 Hz, mono, s16, 176 kb/s
       Stream #0:0(eng): Audio: wmapro (b[1][0][0] / 0x0162), 44100 Hz, stereo, fltp, 128 kb/s
       Stream #0:1: Audio: aac (LC), 48000 Hz, stereo, fltp (default)
       Stream #0:1(eng): Audio: aac (LC), 48000 Hz, stereo, fltp (default)
       Stream #0:0: Audio: wmav2 (a[1][0][0] / 0x0161), 44100 Hz, 2 channels, fltp, 48 kb/s
       Stream #0:1: Audio: ac3, 48000 Hz, 5.1(side), fltp, 384 kb/s (default)
       Stream #0:1[0x1c0]: Audio: mp2, 48000 Hz, mono, s16p, 64 kb/s
       Stream #0:1: Audio: ac3 ([0] [0][0] / 0x2000), 48000 Hz, stereo, fltp, 256 kb/s
       Stream #0:1: Audio: ac3 ([0] [0][0] / 0x2000), 48000 Hz, 5.1(side), fltp, 448 kb/s
       Stream #0:1: Audio: mp3 (U[0][0][0] / 0x0055), 44100 Hz, stereo, s16p, 127 kb/s
       Stream #0:1: Audio: ac3 ([0] [0][0] / 0x2000), 48000 Hz, stereo, fltp, 192 kb/s
       Stream #0:1: Audio: aac (LC), 44100 Hz, stereo, fltp
       Stream #0:0(eng): Audio: wmav2 (a[1][0][0] / 0x0161), 44100 Hz, 2 channels, fltp, 64 kb/s
       Stream #0:1: Audio: mp3, 22050 Hz, stereo, s16p, 32 kb/s
       Stream #0:1: Audio: mp3 (U[0][0][0] / 0x0055), 48000 Hz, stereo, s16p, 128 kb/s

    */

    /**
     * example input:
     * <pre>
     * ", 48 kb/s"
     * ", 256 kb/s"
     * ", 448 kb/s"
     * ", 32 kb/s"
     * ", 384 kb/s (default)"
     * ", 127 kb/s"
     * ", 192 kb/s"
     * ", 64 kb/s"
     * ", 128 kb/s"
     * </pre>
     */
    static final Pattern AUDIO_BYTES_PER_SEC_PATTERN = compile("^\\s*[Ss]tream .* [Aa]udio: .*, ([0-9]+) kb/s\\b");

    /**
     * example input:
     * <pre>
     * ", 5.0"
     * ", 6 channels"
     * ", 0 channels"
     * ", 1 channels"
     * ", mono"
     * ", 2 channels"
     * ", 5.1"
     * ", stereo"
     * </pre>
     */
    static final Pattern AUDIO_CHANNELS_PATTERN = compile("^\\s*[Ss]tream .* [Aa]udio: .*, (mono|stereo|5\\.[10](?:\\([^),]+\\))?|[0-9]+ channels?)");

    /**
     * example input:
     * <pre>
     * "Audio: ac3 (ac-3 / 0x332D6361), "
     * "Audio: pcm_u8 ([1][0][0][0] / 0x0001), "
     * "Audio: eac3, "
     * "Audio: vorbis, "
     * "Audio: aac (HE-AAC) (mp4a / 0x6134706D), "
     * "Audio: mp2 (P[0][0][0] / 0x0050), "
     * "Audio: wmapro (b[1][0][0] / 0x0162), "
     * "Audio: aac (HE-AAC), "
     * "Audio: dts (DTS), "
     * "Audio: mp2, "
     * "Audio: mp3, "
     * </pre>
     */
    static final Pattern AUDIO_CODEC_PATTERN = compile("^\\s*[Ss]tream .* [Aa]udio: ([^,]+?)\\s*,\\s*");

    /**
     * example input:
     * <pre>
     * ", 96000 Hz, "
     * ", 16000 Hz, "
     * ", 11025 Hz, "
     * ", 8000 Hz, "
     * ", 24000 Hz, "
     * ", 32000 Hz, "
     * ", 22050 Hz, "
     * ", 44100 Hz, "
     * ", 48000 Hz, "
     * </pre>
     */
    static final Pattern AUDIO_SAMPLE_RATE_PATTERN = compile("^\\s*[Ss]tream .* [Aa]udio: .*, ([0-9]+) Hz, ");





    @Override
    protected boolean isLineOfInterest(String lineOfMetaData) {
        return LINE_OF_INTEREST_PATTERN.matcher(lineOfMetaData).matches();
    }

    @Override
    protected int parseAudioBytesPerSecond(String lineOfMetaData) {
        return getFirstGroupAsInt(AUDIO_BYTES_PER_SEC_PATTERN.matcher(lineOfMetaData), 1024);
    }

    @Override
    protected String parseAudioChannels(String lineOfMetaData) {
        return getFirstGroupAsString(AUDIO_CHANNELS_PATTERN.matcher(lineOfMetaData));
    }

    @Override
    protected String parseAudioCodec(String lineOfMetaData) {
        return getFirstGroupAsString(AUDIO_CODEC_PATTERN.matcher(lineOfMetaData));
    }

    @Override
    protected int parseAudioSampleRate(String lineOfMetaData) {
        return getFirstGroupAsInt(AUDIO_SAMPLE_RATE_PATTERN.matcher(lineOfMetaData));
    }

    @Override
    protected long parseDurationMillis(String lineOfMetaData) {
        String group = getFirstGroupInternal(DURATION_PATTERN.matcher(lineOfMetaData));
        long result;

        // noinspection SwitchStatementWithTooFewBranches
        switch (group) {
            case NO_MATCH_STRING: {
                result = NO_MATCH_LONG;
                break;
            }
         // case AMBIGUOUS_MATCH_STRING: { // not needed, see also VideoFileParserTest#ensureAllRegexesDefineStartOfInput()
         //     result = AMBIGUOUS_MATCH_LONG;
         //     break;
         // }
            default: {
             // try { // not needed, the pattern ensures it will be numbers

                    String[] parts = group.split("[:.]");

                    int hours = Integer.parseInt(stripLeadingZeros(parts[0]));
                    int minutes = Integer.parseInt(stripLeadingZeros(parts[1]));
                    int seconds = Integer.parseInt(stripLeadingZeros(parts[2]));
                    int hundredths = Integer.parseInt(stripLeadingZeros(parts[3]));

                    long millis = 0L;

                    millis += (1000 * 60 * 60 * hours);
                    millis += (1000 * 60 * minutes);
                    millis += (1000 * seconds);
                    millis += (10 * hundredths);

                    result = millis;

             // } catch (NumberFormatException e) {
             //     result = NOT_A_NUMBER_LONG;
             // }
                break;
            }
        }

        return result;
    }

    private static String stripLeadingZeros(String input) {
        String result = input;
        while (result.charAt(0) == '0' && result.length() > 1) {
            result = result.substring(1);
        }
        return result;
    }

    @Override
    protected int parseTotalBytesPerSecond(String lineOfMetaData) {
        return getFirstGroupAsInt(TOTAL_BYTES_PER_SEC_PATTERN.matcher(lineOfMetaData), 1024);
    }

    @Override
    protected int parseVideoBytesPerSecond(String lineOfMetaData) {
        return getFirstGroupAsInt(VIDEO_BYTES_PER_SECOND_PATTERN.matcher(lineOfMetaData), 1024);
    }

    @Override
    protected long parseVideoStartOffsetMillis(String lineOfMetaData) {
        return getFirstGroupAsLong(VIDEO_START_OFFSET_PATTERN.matcher(lineOfMetaData));
    }

    @Override
    protected String parseVideoCodec(String lineOfMetaData) {
        return getFirstGroupAsString(VIDEO_CODEC_PATTERN.matcher(lineOfMetaData));
    }

    @Override
    protected String parseVideoFormat(String lineOfMetaData) {
        return getFirstGroupAsString(VIDEO_FORMAT_PATTERN.matcher(lineOfMetaData));
    }

    @Override
    protected double parseVideoFrameRate(String lineOfMetaData) {
        return getFirstGroupAsDouble(VIDEO_FRAME_RATE_PATTERN.matcher(lineOfMetaData));
    }

    @Override
    protected int parseVideoHeight(String lineOfMetaData) {
        return getFirstGroupAsInt(VIDEO_HEIGHT_PATTERN.matcher(lineOfMetaData));
    }

    @Override
    protected int parseVideoWidth(String lineOfMetaData) {
        return getFirstGroupAsInt(VIDEO_WIDTH_PATTERN.matcher(lineOfMetaData));
    }


    @Override
    protected List<String> readVideoMetaData(File videoFile2) throws IOException {
        String[] cmds = new String[] {
            getAvconvCommand(),
            "-i", 
            videoFile2.getPath()
        };

        log.trace("          starting process: {}", Util.printArgs(cmds));
        Process p = new ProcessBuilder().redirectErrorStream(true).command(cmds).start();
        BufferedReader input = new BufferedReader(new InputStreamReader(p.getInputStream()));
        List<String> outputLines = new ArrayList<>();
        
        for (String line; (line = input.readLine()) != null;) {
            outputLines.add(line);
            log.trace("          got output line:    > {}", line.trim());
        }

        return outputLines;
    }

}
