package com.myapp.videotools.impl;

import com.myapp.util.file.FileUtils;
import com.myapp.util.format.TimeFormatUtil;
import com.myapp.videotools.AbstractVideoThumbnailer;
import com.myapp.videotools.misc.AppStatistics;
import com.myapp.videotools.misc.Util;
import org.assertj.core.api.Assertions;
import org.assertj.core.util.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.*;
import java.util.stream.Collectors;

import static com.myapp.util.process.ProcessTimeoutKiller.cancelKillTimeout;
import static com.myapp.util.process.ProcessTimeoutKiller.registerKillTimeout;
import static java.util.concurrent.Executors.newFixedThreadPool;
import static org.apache.commons.lang3.StringUtils.leftPad;

public class AvconvVideoThumbnailer extends AbstractVideoThumbnailer implements AvconvConstants {

    private static final Logger LOG = LoggerFactory.getLogger(AvconvVideoThumbnailer.class);

    private static class ThumbnailException extends Exception {
        private ThumbnailException(String message) {
            super(message);
        }
    }

    @Override
    public List<File> createThumbnailSeries(int count, File targetDir) {
        // check preconditions:
        if (videoFile == null) {
            throw new IllegalStateException("videofile must be set!");
        }
        if (count <= 0) {
            throw new IllegalArgumentException("count must be greater 0, but got: " + count);
        }
        LOG.info("        Creating {} thumbnails for:       {}", count, videoFile.getName());
        if (!parseIfNotYetParsed()) {
            LOG.error("        Could not parse video file:       {}", videoFile.getFile());
            return Collections.emptyList();
        }

        LOG.debug("          Length of video file:             {}", TimeFormatUtil.getTimeLabel(videoFile.getLengthSeconds()));
        LOG.debug("          Thumbnails will be created in:    {}", targetDir);


        createOrWipeTempThumbnailDir(targetDir);

        int width = getPreferredWidth();
        int height = getPreferredHeight();

        //-------------------------------------
        // build commands like:
        // ffmpegthumbnailer -i testvideo.mpg -o frame.jpeg -s 270 -t 00:01:36.3
        //-------------------------------------

        List<List<String>> commandSeries = new ArrayList<>();

        for (int i = 0; i < count; i++) {
            List<String> commands = new ArrayList<>();
            commands.add("ffmpegthumbnailer");// TODO: place in configuration file

            // specify input file:
            commands.add("-i");
            commands.add(videoFile.getFile().getPath());

            // specify output file:
            commands.add("-o");
            commands.add(__thumbName(targetDir, i));

            // specify thumb size:
            commands.add("-s");
            String dim = Integer.toString(Math.max(height, width));
            commands.add(dim);

            // specify time offset:
            commands.add("-t");
            commands.add(calculateOffset(i, count));
            commandSeries.add(commands);
        }


        long start = System.currentTimeMillis();

        try {
            __executeThumbnailCmds(commandSeries, targetDir);

        } catch (ThumbnailException te) {
            LOG.error("          "+te.getMessage());
            String message = "at least one thumbnail process did not finish properly. "+te.getMessage();
            throw new RuntimeException(message, te);

        } catch (IOException|InterruptedException e) {
            String message = "an error occured while executing the thumbnail process: " + e;
            LOG.trace("          " + message, e);
            throw new RuntimeException(message, e);

        } finally {
            statistics.addTimeSpentWithThumbnailing(System.currentTimeMillis() - start);
        }


        List<File> results = new ArrayList<>();
        for (int i = 0; i < count; i++) {
            String path = __thumbName(targetDir, i);
            File f = new File(path);
            if (f.isFile()) {
                results.add(f);
                statistics.addThumbnailsCreated(1);
            } else {
                LOG.warn("        FAIL thumbnailing process did not create enough pix: requested: {} file: {}", count, videoFile.getName());
                LOG.warn("        FAIL during command: " + commandSeries.get(i));
                statistics.incrementThumbnailFails();
                return null;
            }
        }

        LOG.info("        OK, {} thumbnails were created.", results.size());

        return results;
    }

    @VisibleForTesting
    protected void __executeThumbnailCmds(List<List<String>> commandSeries, File targetDir)
            throws IOException, InterruptedException, ThumbnailException {

        int cpus = Runtime.getRuntime().availableProcessors();
        ExecutorService es = newFixedThreadPool(Math.max(1, Math.min(commandSeries.size(), cpus)));

        List<Future<Void>> results = commandSeries.stream().map(command -> es.submit((Callable<Void>) () -> {
            ProcessBuilder pb = new ProcessBuilder(command);
            // pb.redirectErrorStream(true);
            int status;
            Process nailProcess = null;

            try {
                LOG.trace("          starting command: " + String.join(" ", command));
                nailProcess = __startProcess(pb);
                registerKillTimeout(nailProcess, 7500, "thumnail_" + targetDir.getName(), true);
                String stdoutLines = Util.getOutput(nailProcess);

                status = __waitForProcess(nailProcess, command);
                LOG.trace("          command finished with status " + status);
                cancelKillTimeout(nailProcess);

                if (status != 0) {
                    throw new ThumbnailException("process " + command + " returned with status " + status + "! output:\n" + stdoutLines);
                }

            } finally {
                if (nailProcess != null)
                    nailProcess.destroy();
                if (LOG.isTraceEnabled()) {
                    LOG.trace("          process 'thumbnails' cleaned up. ({})",
                            targetDir.getName());
                }
            }

            return null;
        })).collect(Collectors.toList());

        es.shutdown();
        es.awaitTermination(commandSeries.size() * 2, TimeUnit.SECONDS);

        for (Future<Void> result : results) {
            try {
                result.get();
            } catch (ExecutionException e) {
                if (e.getCause() instanceof ThumbnailException) {
                    throw (ThumbnailException) e.getCause();
                }
                throw new RuntimeException("expecting only thumbnail exceptions: "+e);
            }
        }
    }

    @VisibleForTesting
    protected static String __thumbName(File targetDir, int num) {
        String name = DEFAULT_THUMB_FILE_NAME_PREFIX + leftPad(""+num, 6, '0') + ".jpeg";
        File file = new File(targetDir, name);
        return file.getPath();
    }

    @Override
    public boolean createBigPicture(File out) {
        LOG.info("      Creating big-picture for video:     {} ...", videoFile.getName());
        LOG.info("        Big-picture target file:          {}", out);

        if (!parseIfNotYetParsed()) {
            LOG.error("        Could not parse video file:       {}", videoFile.getFile());
            return false;
        }

        int rows = getBigPictureRows();
        int cols = getBigPictureCols();
        int width = getPreferredWidth();
        int height = getPreferredHeight();
        final int numberOfThumbsExpected = rows * cols;


        // create the thumbnails first...
        //--------------------------------------

        File tempDir = calculateThumbnailTempDir();
        List<File> thumbs = createThumbnailSeries(numberOfThumbsExpected, tempDir);
        final int thumbsGotReally = thumbs == null ? 0 : thumbs.size();

        if (thumbs == null || thumbs.size() != numberOfThumbsExpected) {
            LOG.error("      FAIL, pictures expected: {}, but got {}",
                    numberOfThumbsExpected, thumbsGotReally);
            return false;
        }


        // ... then merge them to one big-picture ...
        //------------------------------------------------------

        try {
            merger.setRangeSelection(getRangeSelection());
            merger.mergeImages(rows, cols, out, width, height, thumbs, videoFile);

        } catch (Exception e) {
            LOG.error("        Could not merge thumbnails: "+e.getMessage(), e);
            LOG.trace("        Stacktrace:", e);
            LOG.error("      FAIL, no big-picture for video:  {}", videoFile);
            return false;

        } finally {
            FileUtils.deleteRecursively(tempDir);
            LOG.trace("        tmp files removed.");
        }

        LOG.info("      OK, created big-picture for video:  {}", videoFile);
        return true;
    }

    @Override
    public boolean createAnimatedBigPicture(File out, int frameCount) {
        LOG.info("      Creating animated big-picture for:  {} ...", videoFile.getName());
        LOG.info("        Big-picture target file:          {}", out);

        if (!parseIfNotYetParsed()) {
            LOG.error("        Could not parse video file:       {}", videoFile.getFile());
            return false;
        }

        AnimatedBigPictureHelper helper = new AnimatedBigPictureHelper(this);
        helper.animatedBigPic(out, frameCount);

        return true;
    }


    @Override
    public void captureSingleImage(double offset, int width, int height, File out) {
        LOG.info("capturing image of                : " + videoFile.getFile().getPath());
        LOG.info("  target file                     : " + out.getPath());
        LOG.info("  offset in seconds of screenshot : " + TimeFormatUtil.getTimeLabel(offset));

        String[] cmd = createScreenshotCommandArray(offset, width, height, /* true, */ out);
        String output;
        int exitCode;
        Process p = null;
        try {
            p = __startProcess(new ProcessBuilder(cmd));

            if (LOG.isDebugEnabled()) {
                LOG.debug("  started process                : {}", Util.printArgs(cmd));
            }

            registerKillTimeout(p, 1000L, "captureSingleImage_" + out.getName(), false);
            exitCode = __waitForProcess(p, Arrays.asList(cmd));
            output = Util.getOutput(p);
            cancelKillTimeout(p);

        } catch (Exception e) {
            AppStatistics.getInstance().incrementThumbnailFails();
            String msg = "  could not capture image: " + videoFile.getFile().getName();
            LOG.error(msg, e);
            throw new RuntimeException(msg, e);

        } finally {
            if (p != null) {
                p.destroy();
                LOG.trace("  process cleaned up.");
            }
        }
        Assertions.assertThat(exitCode).as(output).isEqualTo(0);
        Assertions.assertThat(out).as(output).isFile();
        Assertions.assertThat(out.length()).as(output).isGreaterThan(1000L);
        AppStatistics.getInstance().incrementThumbnailsCreated();

        LOG.info("OK, image captured to file : {}", out);
    }

    private String[] createScreenshotCommandArray(double offset,
                                                  int pWidth,
                                                  int pHeight,
//                                                  boolean overwriteExisting,
                                                  File outputFile) {
        // "ffmpeg -i /my_video_file_dir/video.flv -y -f image2 -ss 8 -sameq -t 0.001 -s 320*240 /image_dir/screenshot.jpg"
        // 320*240 : image dimension is 320 pixels width and 240 pixels height
        // -ss 8 : screenshot will be taken at 8 second after video starts.

        List<String> args = new ArrayList<>();

        // commandname
        args.add(getAvconvCommand());

        // specify inputfile
        args.add("-i");
        args.add(videoFile.getFile().getPath());

//        if (overwriteExisting) {
            args.add("-y");
//        }

        // set outputformat
        args.add("-f");
        args.add("image2");

        // time of capturing:
        String time = TimeFormatUtil.formatTimeTo2Digits(offset);
        if (LOG.isTraceEnabled()) {
            LOG.trace("formatTime: {}", time);
        }
        args.add("-ss");
        args.add(time);

//        args.add("-sameq");                // see man ffmpeg :-P
        args.add("-t");                    // see man ffmpeg :-P
        args.add("0.001");                 // see man ffmpeg :-P
        args.add("-s");                    // see man ffmpeg :-P
        args.add(pWidth + "*" + pHeight);
        args.add(outputFile.getPath());

        return args.toArray(new String[]{});
    }

    /**
     * calculate the offset for this slice,
     * and format it suitable for the ffmpegthumbnailer command
     * @param num the number of this slice
     * @param count the number of slices
     * @return the offset description string
     */
    private String calculateOffset(int num, int count) {
        long offsetMillis = calculateOffsetMillis(num, count);
        return Util.formatOffset(offsetMillis);
    }

}
