package com.myapp.videotools.impl;

import com.myapp.videotools.IImageMerger;
import com.myapp.videotools.IVideoFileParser;
import com.myapp.videotools.IVideoThumbnailer;
import com.myapp.videotools.misc.Configuration;
import org.assertj.core.util.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Constructor;
import java.util.stream.Stream;


/**
 * @author andre
 *
 */
public final class Application {

    private static final String FFMPEG_PARSER_IMPL_PROPKEY = "com.myapp.videotools.FFMPEG_PARSER_IMPL";
    private static final String THUMBNAILER_IMPL_PROPKEY = "com.myapp.videotools.THUMBNAILER_IMPL";
    private static final String IMAGE_MERGER_IMPL_PROPKEY = "com.myapp.videotools.IMAGE_MERGER_IMPL";
    
    private static final Logger LOG = LoggerFactory.getLogger(Application.class);

    
    private static Application instance = null;


    public static Application getInstance() {
        if (instance == null)
            synchronized (Application.class) {
                if (instance == null)
                    instance = new Application();
            }

        return instance;
    }

    

    private Application() {
        LOG.debug("Application wrapper instance created!");
    }

    public IVideoFileParser createVideoFileParser() {
        String className = Configuration.getInstance().getProperty(FFMPEG_PARSER_IMPL_PROPKEY);
        Object o = getInstance(className);
        LOG.debug("  IVideoFileParser impl is:     {}", o.getClass());
        return (IVideoFileParser) o;
    }
    
    public IVideoThumbnailer createVideoThumbnailer() {
        String className = Configuration.getInstance().getProperty(THUMBNAILER_IMPL_PROPKEY);
        Object o = getInstance(className);
        LOG.debug("  IVideoThumbnailer impl is:    {}", o.getClass());
        return (IVideoThumbnailer) o;
    }
    
    public IImageMerger createImageMerger() {
        String className = Configuration.getInstance().getProperty(IMAGE_MERGER_IMPL_PROPKEY);
        Object o = getInstance(className);
        LOG.debug("  IImageMerger impl is:         {}", o.getClass());
        return (IImageMerger) o;
    }

    @VisibleForTesting
    static Object getInstance(String classname) {
        Class<?> implClass;
        
        try {
            implClass = Class.forName(classname);
            
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("no such class :"+classname+" !", e);
        }
        
        Constructor<?>[] constructors = implClass.getConstructors();
        Constructor<?> defaultConstructor = Stream.of(constructors)
                .filter(c -> c.getParameterTypes().length == 0)
                .findAny()
                .orElseThrow(() -> new RuntimeException("no default constructor for: "+classname));

        try {
            return defaultConstructor.newInstance();
        } catch (Exception e) {
            throw new RuntimeException("could not create "+classname+" !", e);
        }
    }

}
