package com.myapp.videotools.impl;

import com.myapp.util.image.ImageUtil;
import com.myapp.videotools.AbstractVideoThumbnailer;
import com.myapp.videotools.IImageMerger;
import com.myapp.videotools.VideoFile;
import com.myapp.videotools.misc.Configuration;
import com.myapp.videotools.misc.Util;
import org.assertj.core.api.Assertions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.awt.*;
import java.io.File;
import java.util.List;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import static java.util.concurrent.Executors.newFixedThreadPool;
import static org.apache.commons.io.FileUtils.*;

/**
 * encapsulates creation of animated pictures
 */
class AnimatedBigPictureHelper {

    class Burst implements Callable<List<File>> {
        long offsetMillis;
        int frameCount;
        Dimension frameSize;

        File inputFile;
        File workingDir;
        String marker;
        List<File> collectedFiles;

        @Override
        public List<File> call() throws Exception {
            deleteQuietly(workingDir);
            forceMkdir(workingDir);

            String ffmpeg = Configuration.getInstance().getProperty(AvconvConstants.FFMPEG_COMMAND_PROPKEY);
            final List<String> command = Arrays.asList(
                    ffmpeg, "-i", inputFile.getPath(),
                    "-ss", Util.formatOffset(offsetMillis),
                    "-frames", String.valueOf(frameCount),
                    "-s", frameSize.width + "x" + frameSize.height,
                    workingDir.getPath() + File.separator + "frame-%003d.jpeg"
            );

            LOG.debug(marker + " Starting process for: " + String.join(" ", command));

            int exitCode;
            String output;
            try {
                Process process = thumbnailer.__startProcess(new ProcessBuilder().command(command));
                exitCode = thumbnailer.__waitForProcess(process, command);
                output = Util.getOutput(process);
            } catch (Exception e) {
                throw new RuntimeException("process for " + marker + " threw: " + e.getClass().getSimpleName(), e);
            }
            if (exitCode != 0) {
                throw new RuntimeException("process for " + marker + " failed, exitCode was " + exitCode + "!" +
                        "\n" + output);
            }

            collectedFiles = listFiles(workingDir, null, false)
                    .stream()
                    .sorted(new Util.DirsFirstAlphabeticFileComparator())
                    .peek(f -> Assertions.assertThat(f).isFile())
                    .collect(Collectors.toList());

            LOG.debug(marker + " Process finished, frames captured for burst: " + collectedFiles.size());

            return collectedFiles;
        }
    }

    private static final Logger LOG = LoggerFactory.getLogger(AnimatedBigPictureHelper.class);

    private final AbstractVideoThumbnailer thumbnailer;
    private VideoFile videoFile;
    private IImageMerger merger;

    AnimatedBigPictureHelper(AbstractVideoThumbnailer thumbnailer) {
        this.thumbnailer = Objects.requireNonNull(thumbnailer);
        this.merger = Application.getInstance().createImageMerger();
    }

    @SuppressWarnings("WeakerAccess")
    public void animatedBigPic(File out, int frameCount) {
        this.videoFile = Objects.requireNonNull(thumbnailer.getVideoFile());

        File tempDir = thumbnailer.calculateThumbnailTempDir();
        thumbnailer.createOrWipeTempThumbnailDir(tempDir);

        try {

            // step 1: create a set of frames for each tile in the grid - so called bursts.


            // a burst is a number of consecutive frames captured from a video file at a given offset.
            List<Burst> bursts = createBurstFrames(frameCount, tempDir);


            // step 2: create multiple bigpics.


            // every bigpic is created from the same frame of each burst.
            // afterwards we can generate a gif-bigpic out of these single bigpics.
            createMultipleBigPicsFromBursts(frameCount, tempDir, bursts);


            // step 3: create a gif from the generated single jpeg bigpics:


            createGifFromMultipleBigPics(out, tempDir);

        } finally {
            deleteQuietly(tempDir);
        }
    }

    private List<Burst> createBurstFrames(int frameCount, File tempDir) {
        int rows = thumbnailer.getBigPictureRows();
        int cols = thumbnailer.getBigPictureCols();

        Dimension dimension = ImageUtil.scaleDimensions(
                thumbnailer.getPreferredWidth(), thumbnailer.getPreferredHeight(),
                videoFile.getVideoWidth(), videoFile.getVideoHeight());

        List<Burst> bursts = new ArrayList<>();
        for (int i = 0, r = 0; r < rows; r++) {
            for (int c = 0; c < cols; c++, i++) {
                Burst burst = new Burst();
                burst.workingDir = new File(tempDir, "burst_" + (i < 10 ? "00" : i < 100 ? "0" : "") + i + "_r" + r + "c" + c);
                burst.marker = burst.workingDir.getName() + " ";
                burst.inputFile = videoFile.getFile();
                burst.frameSize = dimension;
                burst.frameCount = frameCount;
                burst.offsetMillis = thumbnailer.calculateOffsetMillis(i, rows * cols);
                bursts.add(burst);
            }
        }

        Map<Burst, Future<List<File>>> futureMap = new LinkedHashMap<>();

        int cpus = Runtime.getRuntime().availableProcessors();
        ExecutorService es1 = newFixedThreadPool(Math.max(1, Math.min(rows * cols, cpus)));
        bursts.forEach(task -> futureMap.put(task, es1.submit(task)));
        try {
            thumbnailer.__awaitTermination(es1, 500L * rows * cols * frameCount, "create burst frames");
        } catch (InterruptedException e) {
            throw new RuntimeException("an error occued during creating bursts", e);
        }

        futureMap.forEach((burst, futureFiles) -> {
            try {
                futureFiles.get();
            } catch (Exception e) {
                throw new RuntimeException("error during creating burst with offset " +
                        burst.offsetMillis, e);
            }
        });
        bursts.forEach(b -> Assertions.assertThat(b.collectedFiles).hasSize(frameCount));
        return bursts;
    }

    private void createMultipleBigPicsFromBursts(int frameCount, File tempDir, List<Burst> bursts) {
        int thumbWidth2 = thumbnailer.getPreferredWidth();
        int thumbHeight2 = thumbnailer.getPreferredHeight();
        int rows2 = thumbnailer.getBigPictureRows();
        int cols2 = thumbnailer.getBigPictureCols();

        int cpus2 = Runtime.getRuntime().availableProcessors();
        ExecutorService es2 = newFixedThreadPool(Math.max(1, Math.min(frameCount, cpus2)));
        Map<Integer, File> bigPics = new TreeMap<>();

        IntStream.range(0, frameCount).mapToObj(i -> (Callable<File>) () -> {
            int k = i + 1;
            String oneBasedIdx = "" + k;
            File bigPic = new File(tempDir, "bigPic_" + oneBasedIdx + ".jpeg");
            bigPics.put(i, bigPic);
            List<File> files = bursts.stream()
                    .map(b -> b.collectedFiles.get(i))
                    .collect(Collectors.toList());

            deleteQuietly(bigPic);
            merger.mergeImages(rows2, cols2, bigPic, thumbWidth2, thumbHeight2, files, videoFile);
            Assertions.assertThat(bigPic).isFile();
            return bigPic;

        }).forEach(es2::submit);

        try {
            thumbnailer.__awaitTermination(es2, 1500L * frameCount, "create bigpics from bursts");
        } catch (InterruptedException e) {
            throw new RuntimeException("an error occured while creating bigpics from bursts", e);
        }

        Assertions.assertThat(bigPics).hasSize(frameCount);
        bigPics.forEach((k, v) -> Assertions.assertThat(v).isFile());
    }


    private void createGifFromMultipleBigPics(File out, File tempDir) {
        String ffmpeg = Configuration.getInstance().getProperty(AvconvConstants.FFMPEG_COMMAND_PROPKEY);
        // ffmpeg -i bigPic_%0004d.jpeg -f gif video2.gif
        List<String> command = Arrays.asList(
                ffmpeg, "-i", tempDir.getPath() + File.separator + "bigPic_%d.jpeg",
                "-f", "gif",
                out.getPath()
        );

        LOG.debug("Starting gif merge process for: " + String.join(" ", command));
        int exitCode;
        String output;
        try {
            Process process = thumbnailer.__startProcess(new ProcessBuilder().command(command));
            exitCode = thumbnailer.__waitForProcess(process, command);
            output = Util.getOutput(process);
        } catch (Exception e) {
            throw new RuntimeException("gif merge process threw: " + e.getClass().getSimpleName(), e);
        }
        if (exitCode != 0) {
            throw new RuntimeException("gif merge process for " + out + " failed, exitCode was " + exitCode + "!" +
                    "\n" + output);
        }
        Assertions.assertThat(exitCode).as(output).isEqualTo(0);
        Assertions.assertThat(out).as(output).isFile();
        Assertions.assertThat(out.length()).as(output).isGreaterThan(1000L);

        LOG.debug("Gif merged to: " + out.getPath());
    }

}
