package com.myapp.videotools.commandline;

import com.myapp.videotools.VideoFile;
import org.apache.commons.lang3.StringUtils;
import picocli.CommandLine.Command;
import picocli.CommandLine.ITypeConverter;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

@Command
@SuppressWarnings("WeakerAccess")
public class RangeSelection {

    public static final double DEFAULT_RATIO = 1.0;
    public static final long DEFAULT_OFFSET = 0L;

    public static final RangeSelection COMPLETELY = new RangeSelection() {
        @Override
        public Coords calculate(VideoFile videoFile) {
            checkParsed(videoFile);
            long lengthMillis = videoFile.getLengthMillis();
            return new Coords(0, lengthMillis);
        }
    };

    public static class Coords {
        /**
         * how long is the selected clip?
         */
        private final long lengthMillis;
        /**
         * where does the selected clip start?
         */
        private final long offsetMillis;

        Coords(long offsetMillis, long lengthMillis) {
            this.offsetMillis = offsetMillis;
            this.lengthMillis = lengthMillis;
        }

        public long getOffsetMillis() {
            return offsetMillis;
        }

        public long getLengthMillis() {
            return lengthMillis;
        }
    }


    @SuppressWarnings("JavadocReference")
    public static class Converter implements ITypeConverter<RangeSelection> {

        private static final Pattern
                OFFSET_ONLY = Pattern.compile("(?x) ^ @ \\s* ([-+]?) \\s* (\\d+) $"),
                PERCENT_ONLY = Pattern.compile("(?x) ^ ([-+]?) \\s* (\\d+ \\.? \\d*) \\s* [%] $"),
                OFFSET_AND_PERCENT = Pattern.compile("(?x) ^ [+]? \\s* (\\d+ \\.? \\d*) \\s* [%] \\s* @ \\s* [+]? \\s* (\\d+) $")
//                PERCENT_AND_PERCENT = Pattern.compile("x")// TODO: offset in percent and duration in percent
        ;


        /**
         * @see RangeSelectionTest
         */
        @Override
        public RangeSelection convert(String value) {
            String s = StringUtils.trimToEmpty(value);
            if (StringUtils.isEmpty(s)) {
                return RangeSelection.COMPLETELY;
            }

            Matcher matcher;

            if ((matcher = PERCENT_ONLY.matcher(s)).matches()) {
                String sign = StringUtils.trimToEmpty(matcher.group(1));
                String percentString = StringUtils.trimToEmpty(matcher.group(2));
                double percentValue = Double.parseDouble(percentString);

                return "-".equals(sign)
                        ? RangeSelection.ofLastPercent(percentValue)
                        : RangeSelection.ofFirstPercent(percentValue);
            }

            if ((matcher = OFFSET_ONLY.matcher(s)).matches()) {
                String sign = StringUtils.trimToEmpty(matcher.group(1));
                String offsetMillisString = StringUtils.trimToEmpty(matcher.group(2));
                long offset = Long.parseLong(offsetMillisString);

                RangeSelection rangeSelection = new RangeSelection();
                rangeSelection.setOffsetMillis("-".equals(sign) ? -offset : offset);
                rangeSelection.setLengthRatio(null);
                return rangeSelection;
            }

            if ((matcher = OFFSET_AND_PERCENT.matcher(s)).matches()) {
                String percentString = StringUtils.trimToEmpty(matcher.group(1));
                String offsetMillisString = StringUtils.trimToEmpty(matcher.group(2));

                double percentValue = 0.01 * Double.parseDouble(percentString);
                long offset = Long.parseLong(offsetMillisString);

                RangeSelection rangeSelection = new RangeSelection();
                rangeSelection.setOffsetMillis(offset);
                rangeSelection.setLengthRatio(percentValue);
                return rangeSelection;
            }

            throw new IllegalArgumentException("Unsupported Range Selection format: '" + s + "'");
        }
    }

    private Double lengthRatio = DEFAULT_RATIO;
    private Long offsetMillis = DEFAULT_OFFSET;

    public void setLengthRatio(Double lengthRatio) {
        if (lengthRatio == null) {
            this.lengthRatio = lengthRatio;
            return;
        }

        if (lengthRatio > -0.00001 && lengthRatio < 0.00001) {
            throw new IllegalArgumentException("ratio zero is not allowed.");
        }
        if (lengthRatio > 1.0 || lengthRatio < -1.0) {
            throw new IllegalArgumentException("ratio not within -1 and 1: " + lengthRatio);
        }

        this.lengthRatio = lengthRatio;
    }

    public void setOffsetMillis(Long offsetMillis) {
        this.offsetMillis = offsetMillis;
    }

    public static RangeSelection ofFirstPercent(double percent) {
        RangeSelection partSelection = new RangeSelection();
        partSelection.setOffsetMillis(null);
        partSelection.setLengthRatio(percent * 0.01);
        return partSelection;
    }

    public static RangeSelection ofLastPercent(double percent) {
        RangeSelection partSelection = new RangeSelection();
        partSelection.setOffsetMillis(null);
        partSelection.setLengthRatio(percent * -0.01);
        return partSelection;
    }

    public Coords calculate(VideoFile videoFile) {
        checkParsed(videoFile);
        long totalDuration = videoFile.getLengthMillis();

        if (offsetMillis != null) {
            if (lengthRatio == null) {

                // if there is only an offset given and no ratio, just cut off the
                // offset at the start (or at the end if negative) :

                long offset;
                long duration;

                if (offsetMillis < 0) {
                    duration = 0 - offsetMillis;
                    offset = totalDuration - duration;
                } else {
                    duration = totalDuration - offsetMillis;
                    offset = offsetMillis;
                }

                return new Coords(offset, duration);
            }

            // if there is an offset and a ratio given,
            // use the n% part from the given offset.
            // negative references are not supported.

            long offset = offsetMillis;
            long duration = Math.round(lengthRatio * totalDuration);
            return new Coords(offset, duration);
        }



        // if there is only a ratio given and no offset,
        // use the first n% from the start (or from the end if negative) :
        if (lengthRatio != null) {
            long offset = 0L;
            long duration = Math.round(lengthRatio * totalDuration);

            if (lengthRatio < 0) { // select the last part of the video
                duration = - duration;
                offset = totalDuration - duration;
            }

            return new Coords(offset, duration);
        }

        // if neither an offset nor a ratio is given,
        // fallback to the entire video length :
        return new Coords(0L, totalDuration);
    }

    void checkParsed(VideoFile videoFile) {
        if (!videoFile.isParsed()) {
            throw new IllegalStateException("only parsed videofiles please.");
        }
    }

    public Double getLengthRatio() {
        return lengthRatio;
    }

    public Long getOffsetMillis() {
        return offsetMillis;
    }
}
