package com.myapp.videotools.commandline;

import com.myapp.util.text.NumericStringComparator;
import com.myapp.videotools.misc.Broadcast;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;
import picocli.custom.CustomTypeConverters.PositiveInteger;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static com.myapp.videotools.commandline.CheckForDownloadableMoviesCommand.*;
import static com.myapp.videotools.commandline.Filters.FilterType.*;
import static picocli.CommandLine.Help.Visibility.ALWAYS;

/**
 * Encapsulates Filter logic for Broadcast Objects.
 *
 * @see CheckForDownloadableMoviesCommand
 * @author andre
 */
@Command
public class Filters {

    enum FilterType {
        TITLE_EXACT,
        TITLE_PREFIX,
        TITLE_REGEX,
        CHANNEL,
        TIME,
        AD_COPY,
        DESCRIPTION_REGEX,
        PARSE_ERROR
    }

    static class FilterResult {
        String reason;
        FilterType type;

        FilterResult(FilterType type, String reason) {
            this.type = type;
            this.reason = reason;
        }

        FilterResult(FilterType type) {
            this(type, null);
        }
    }

    @SuppressWarnings("WeakerAccess")
    public static final int DEFAULT_THRESHOLD_SECS = 30 * 60;

    @SuppressWarnings("WeakerAccess")
    public static final String OPT_NON_MATCHING_FILTERS = "--show-non-matching-filters";

    private Set<String> blackListPrefix = new TreeSet<>();
    private Set<String> blackList = new TreeSet<>();
    private Set<String> blackListChannels = new TreeSet<>();
    private Set<String> blackListRegex = new TreeSet<>();
    private Set<String> blackListDescriptionRegex = new TreeSet<>();


    @Option(names = {OPT_THRESHOLD, "-dth"},
            converter = PositiveInteger.class,
            defaultValue = "" + DEFAULT_THRESHOLD_SECS,
            showDefaultValue = ALWAYS,
            paramLabel = "INT",
            description = "The minimum duration in seconds of a tvthek item, shorter items will be filtered.")
    private int thresholdSecs = DEFAULT_THRESHOLD_SECS;

    @Option(names = {OPT_NON_MATCHING_FILTERS, "-showfilt"},
            showDefaultValue = ALWAYS,
            defaultValue = "false", // TODO: set to false when this command is ready developed
            description = "Shows which filters did not match any broadcasts.")
    private boolean showUnNecessaryFilters = false;

    private Map<FilterType, List<FilterResult>> stats = new EnumMap<>(FilterType.class);

    public Filters() {
        populateBlacklists();
    }

    public void clear() {
        stats.clear();
        blackListChannels.clear();
        blackListPrefix.clear();
        blackListRegex.clear();
        blackList.clear();
        blackListDescriptionRegex.clear();
    }

    private List<FilterResult> filter(Broadcast bc, List<Broadcast> allBroadcasts) {
        List<FilterResult> matchingFilters = new ArrayList<>();
        if (bc == null) {
            matchingFilters.add(new FilterResult(PARSE_ERROR));
            return matchingFilters;
        }

        String title = bc.getTitle().replaceAll("^AD \\| ", "");

        if (blackList.contains(title)) {
            matchingFilters.add(new FilterResult(TITLE_EXACT, title));
        }

        blackListPrefix.forEach(prefix -> {
            if (title.startsWith(prefix)) {
                matchingFilters.add(new FilterResult(TITLE_PREFIX, prefix));
            }
        });

        blackListRegex.forEach(regex -> {
            if (Pattern.compile(regex).matcher(title).find()) {
                matchingFilters.add(new FilterResult(TITLE_REGEX, regex));
            }
        });
        blackListDescriptionRegex.forEach(regex -> {
            if (Pattern.compile(regex).matcher(bc.getDescription()).find()) {
                matchingFilters.add(new FilterResult(DESCRIPTION_REGEX, regex));
            }
        });

        if (bc.getTitle().startsWith("AD | ")
                && allBroadcasts.stream().anyMatch(x -> x.getTitle().equals(title))) {
            matchingFilters.add(new FilterResult(AD_COPY, bc.getTitle()));
        }

        if (bc.getDurationSeconds() > 0 && bc.getDurationSeconds() < thresholdSecs) {
            matchingFilters.add(new FilterResult(TIME));
        }

        if (blackListChannels.contains(bc.getTvChannel())) {
            matchingFilters.add(new FilterResult(CHANNEL, bc.getTvChannel()));
        }

        return matchingFilters;
    }

    @SuppressWarnings("WeakerAccess")
    public void showUnnecessaryFilters() {
        Arrays.stream(FilterType.values())
                .filter(key -> !stats.containsKey(key))
                .forEach(x -> System.err.println("Filtertype with no match: " + x));

        List<FilterResult> byTitleExact = stats.get(TITLE_EXACT);
        if (byTitleExact != null) {
            blackList.stream().sorted()
                    .filter(title -> byTitleExact.stream().noneMatch(hit -> title.equals(hit.reason)))
                    .forEach(x -> System.err.println("blackList entry '"+x+"' did not match anything."));
        }

        List<FilterResult> byChannel = stats.get(CHANNEL);
        if (byChannel != null) {
            blackListChannels.stream().sorted()
                    .filter(channel -> byChannel.stream().noneMatch(hit -> channel.equals(hit.reason)))
                    .forEach(x -> System.err.println("blackListChannel entry '"+x+"' did not match anything."));
        }

        List<FilterResult> byTitleRegex = stats.get(TITLE_REGEX);
        if (byTitleRegex != null) {
            blackListRegex.stream().sorted()
                    .filter(regex -> byTitleRegex.stream().noneMatch(hit -> regex.equals(hit.reason)))
                    .forEach(x -> System.err.println("blackListRegex entry '"+x+"' did not match anything."));
        }

        List<FilterResult> byTitlePrefix = stats.get(TITLE_PREFIX);
        if (byTitlePrefix != null) {
            blackListPrefix.stream()
                    .filter(channel -> byTitlePrefix.stream().noneMatch(hit -> channel.equals(hit.reason)))
                    .sorted()
                    .forEach(x -> System.err.println("blackListPrefix entry '"+x+"' did not match anything."));
        }

        List<FilterResult> byDescriptionRegex = stats.get(DESCRIPTION_REGEX);
        if (byDescriptionRegex != null) {
            blackListDescriptionRegex.stream()
                    .filter(channel -> byDescriptionRegex.stream().noneMatch(hit -> channel.equals(hit.reason)))
                    .sorted()
                    .forEach(x -> System.err.println("descriptionRegex entry '"+x+"' did not match anything."));
        }

        for (String p : blackListPrefix) {
            for (String s : blackList) {
                if (s.startsWith(p)) {
                    System.err.println(s + " starts with " + p);
                }
            }
        }
        for (String p : blackListRegex) {
            Matcher matcher = Pattern.compile(p).matcher("foo");

            for (String s : blackList) {
                if (matcher.reset(s).find()) {
                    System.err.println(s + " matches " + p);
                }
            }
            for (String s : blackListPrefix) {
                if (matcher.reset(s).find()) {
                    System.err.println(s + " matches " + p);
                }
            }
        }
    }

    @SuppressWarnings("WeakerAccess")
    public String showFilterStatistics() {
        List<FilterResult> byTime = stats.get(TIME);
        List<FilterResult> byTitleExact = stats.get(TITLE_EXACT);
        List<FilterResult> byChannel = stats.get(CHANNEL);
        List<FilterResult> byTitleRegex = stats.get(TITLE_REGEX);
        List<FilterResult> byAdCopy = stats.get(AD_COPY);
        List<FilterResult> byTitlePrefix = stats.get(TITLE_PREFIX);
        List<FilterResult> byDescriptionRegex = stats.get(DESCRIPTION_REGEX);
        List<FilterResult> byParseError = stats.get(PARSE_ERROR);

        List<String> stmts = new ArrayList<>();
        if (byTime != null) {
            stmts.add("Time('t < " + (thresholdSecs / 60) + " min':" + byTime.size() + ")");
        }
        if (byTitleExact != null) {
            stmts.add("Title(" + byTitleExact.size() + ")");
        }
        if (byChannel != null) {
            stmts.add("Channel(" + distinctCountsString(byChannel) + ")");
        }
        if (byAdCopy != null) {
            stmts.add("AdCopy(" + byAdCopy.size() + ")");
        }
        if (byTitleRegex != null) {
            stmts.add("TitleRegex(" + distinctCountsString(byTitleRegex) + ")");
        }
        if (byTitlePrefix != null) {
            stmts.add("TitlePrefix(" + distinctCountsString(byTitlePrefix) + ")");
        }
        if (byDescriptionRegex != null) {
            stmts.add("DescriptionRegex(" + distinctCountsString(byDescriptionRegex) + ")");
        }
        if (byParseError != null) {
            stmts.add("Unparseable(" + byParseError.size() + ")");
        }
        return "FilterType: [\n    " + String.join("\n    ", stmts) + "\n]";
    }

    @SuppressWarnings("WeakerAccess")
    public boolean isInteresting(Broadcast bc, List<Broadcast> allBroadcasts) {
        return 0 == filter(bc, allBroadcasts)
                .stream()
                .peek(fr -> stats.computeIfAbsent(fr.type, foo -> new ArrayList<>()).add(fr))
                .count();
    }

    private static String distinctCountsString(List<FilterResult> filterResults) {
        return filterResults.size() + " - " +  filterResults.stream()
                .map(x -> x.reason)
                .distinct()
                .collect(Collectors.toMap(
                        channelName -> channelName,
                        cn -> filterResults.stream().filter(x -> x.reason.equals(cn)).count()))
                .entrySet()
                .stream()
                .map(e -> e.getValue() + "x'" + e.getKey() + "'")
                .sorted(new NumericStringComparator().reversed())
                .collect(Collectors.joining(", "));
    }

    @SuppressWarnings("WeakerAccess")
    public boolean isShowUnNecessaryFilters() {
        return showUnNecessaryFilters;
    }

    @SuppressWarnings("WeakerAccess")
    public void setShowUnNecessaryFilters(boolean showUnNecessaryFilters) {
        this.showUnNecessaryFilters = showUnNecessaryFilters;
    }

    private void populateBlacklists() {
        blackListChannels.add("ORF Sport+");

        blackListRegex.add("(?i)wahlkampf");
        blackListRegex.add("\\b(SPÖ|FPÖ|ÖVP)\\b");
        blackListRegex.add("(?i)\\b(Weltmeisterschaft|Weltcup|Nordische Kombination|Ski alpin)\\b");
        blackListRegex.add("\\b(Weltcup)\\b");

        blackListPrefix.add("Pressefoyer ");
        blackListPrefix.add("Pressegespräch ");
        blackListPrefix.add("Pressekonferenz: ");
        blackListPrefix.add("Pressekonferenz ");
        blackListPrefix.add("Rosamunde Pilcher: ");
        blackListPrefix.add("Fußball: ");
        blackListPrefix.add("Biathlon-WM ");
        blackListPrefix.add("ZIB ");
        blackListPrefix.add("Guten Morgen Österreich");
        blackListPrefix.add("Wetter");
        blackListPrefix.add("Politik Spezial: ");
        blackListPrefix.add("Snowboard: ");
        blackListPrefix.add("Tischtennis: ");
        blackListPrefix.add("Judo: ");
        blackListPrefix.add("Rallye: ");
        blackListPrefix.add("Weltjournal");
        blackListPrefix.add("Sport ");
        blackListPrefix.add("Fußball");
        blackListPrefix.add("Sport-");
        blackListPrefix.add("Stöckl ");
        blackListPrefix.add("STÖCKL.");
        blackListPrefix.add("Tatort");
        blackListPrefix.add("Seitenblicke");
        blackListPrefix.add("[WETTER] ZIB");
        blackListPrefix.add("Dancing Stars");
        blackListPrefix.add("Lasko ");
        blackListPrefix.add("Mordshunger ");
        blackListPrefix.add("Herzflimmern ");
        blackListPrefix.add("Nationalrat");
        blackListPrefix.add("ORF III Aktuell");
        blackListPrefix.add("Medicopter 117");
        blackListPrefix.add("Bürgeranwalt");
        blackListPrefix.add("Die Chefin");

        for (String x : new String[]{"Österreich", "Vorarlberg", "Tirol", "Salzburg", "Kärnten",
                "Steiermark", "Oberösterreich", "Niederösterreich", "Wien", "Burgenland", "Südtirol",}) {
            blackListPrefix.add(x + " heute");
        }

        blackList.add("Spät-ZIB");
        blackList.add("Soko Kitzbühel");
        blackList.add("Servus Kasperl");
        blackList.add("Newton");
        blackList.add("Report");
        blackList.add("Thema");
        blackList.add("Themenmontag");
        blackList.add("Kommissar Rex");
        blackList.add("Die Millionenshow");
        blackList.add("Euromillionen");
        blackList.add("Soko Donau");
        blackList.add("Lotto 6 aus 45");
        blackList.add("Helmi");
        blackList.add("Eco");
        blackList.add("Hohes Haus");
        blackList.add("Heimat Österreich");
        blackList.add("Land der Berge");
        blackList.add("Im Brennpunkt");
        blackList.add("Im Zentrum");
        blackList.add("Formel 1");
        blackList.add("konkret");
        blackList.add("konkret (ÖGS)");
        blackList.add("kreuz und quer");
        blackList.add("Tom Turbo");
        blackList.add("Unser Österreich");
        blackList.add("Unterwegs in Österreich");
        blackList.add("Barbara Karlich Show");
        blackList.add("České & Slovenské Ozveny");
        blackList.add("Dobar dan Hrvati");
        blackList.add("Dober dan, Koroška");
        blackList.add("Dober dan, Štajerska");
        blackList.add("Echt Fett");
        blackList.add("Schmatzo - Der Koch-Kids-Club");
        blackList.add("Schmatzo - Kochen mit WOW");
        blackList.add("Servus Szia Zdravo Del tuha");
        blackList.add("Servus, Srečno, Ciao");
        blackList.add("Willkommen Österreich");
        blackList.add("Mein Döbling");
        blackList.add("Klingendes Österreich");
        blackList.add("Verstehen Sie Spaß?");
        blackList.add("zeit.geschichte");
        blackList.add("Mittag in Österreich");
        blackList.add("Mord in bester Gesellschaft");
        blackList.add("Pressestunde");
        blackList.add("Erlebnis Bühne");
        blackList.add("Wilde Reise mit Erich Pröll");
        blackList.add("erLesen");
        blackList.add("kulturMontag");
        blackList.add("matinee am Sonntag");
        blackList.add("Politik live");
        blackList.add("DOK.eins");
        blackList.add("Expeditionen");
        blackList.add("Studio 2");
        blackList.add("Universum History");
        blackList.add("Universum");
        blackList.add("Am Schauplatz");
        blackList.add("Treffpunkt Medizin");
        blackList.add("Der Zigeunerbaron");
        blackList.add("Der Alte");
        blackList.add("Die Tafelrunde");
        blackList.add("Was gibt es Neues?");
        blackList.add("Was schätzen Sie?");
        blackList.add("Doctor's Diary");
        blackList.add("Bürgeranwalt");
        blackList.add("Erbe Österreich");
        blackList.add("MERYNS sprechzimmer");
        blackList.add("Quantensprung");
        blackList.add("Evangelischer Gottesdienst");
        blackList.add("Rosamunde Pilcher");
        blackList.add("ORF-1-Freistunde: ");
        blackList.add("Q1 Ein Hinweis ist falsch");
        blackList.add("AKTUELL nach eins");
        blackList.add("Bergweihnacht");
        blackList.add("Die Promi-Millionenshow");
        blackList.add("Die Wintercamper");
        blackList.add("Vera");
        blackList.add("Katholischer Gottesdienst");
        blackList.add("Quiz ohne Grenzen");
        blackList.add("Talk 1");
        blackList.add("matinee am Feiertag");
        blackList.add("Das Traumhotel");
        blackList.add("Letzter Wille");
        blackList.add("Gute Nacht Österreich");
        blackList.add("xxxxxxxxxxxxxxxxxxxx");
        blackList.add("xxxxxxxxxxxxxxxxxxxx");
        blackList.add("xxxxxxxxxxxxxxxxxxxx");
        blackList.add("xxxxxxxxxxxxxxxxxxxx");
        blackList.add("xxxxxxxxxxxxxxxxxxxx");
        blackList.add("xxxxxxxxxxxxxxxxxxxx");
        blackList.add("xxxxxxxxxxxxxxxxxxxx");

        blackListDescriptionRegex.add("Polit-Talk");
        blackListDescriptionRegex.add("Krimiserie");
    }

}
