package com.myapp.videotools.commandline;


import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import picocli.CommandLine;
import picocli.CommandLine.*;
import picocli.CommandLine.Help.Ansi;

import java.io.PrintStream;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;

import static com.myapp.videotools.commandline.VideoToolsBaseCommand.*;
import static picocli.CommandLine.Help.Ansi.AUTO;
import static picocli.CommandLine.Help.defaultColorScheme;

/**
 * Main handler for the video tools collection.
 * Parses and interprets the passed command line arguments.
 */
@Command(subcommands = {
        CommandLine.HelpCommand.class,
        BigPictureCommand.class,
        CaptureFrameCommand.class,
        CreateThumbsCommand.class})
public class CommandLineInterpreter {

    private static final Logger LOGGER = LoggerFactory.getLogger(CommandLineInterpreter.class);


    public static int runProgram(PrintStream out, PrintStream err, String... args) {
        if (args == null) {
            // programmatic error, cannot happen when jvm starts main() method.
            throw new NullPointerException("args is null");
        }

        CommandLine testCommand = new CommandLine(new CommandLineInterpreter());
        if (args.length == 0) {
            showUsage(err, testCommand);
            return 1;
        }

        if (args[0].equals(HELP) || args[0].equals(HELP2) || args[0].equals(HELP3)) {
            showUsage(out, testCommand);
            return 0;
        }

        AtomicInteger exitCode = new AtomicInteger(0);
        DefaultExceptionHandler<List<Object>> eh = new DefaultExceptionHandler<List<Object>>() {
            @Override
            public List<Object> handleException(ParameterException ex,
                                                PrintStream out, Ansi ansi, String... args) {
                return handleParseException(ex, args);
            }
            @Override
            public List<Object> handleExecutionException(ExecutionException ex, ParseResult pr) {
                exitCode.set(2);
                String errorMessage = ex.getMessage();
                for (Throwable cause = ex; cause != null; cause = cause.getCause()) {
                    if (!(cause instanceof PicocliException)) {
                        errorMessage = cause.getMessage();
                        break;
                    }
                }
                err.println("ERROR: " + errorMessage);
                LOGGER.debug("An error occured during execution.", ex);
                return null;
            }
            @Override
            public List<Object> handleParseException(ParameterException ex, String[] args) {
                exitCode.set(1);
                err.println("ERROR: " + ex.getMessage());
                LOGGER.debug("An error occured during parsing " + Arrays.toString(args), ex);
                return null;
            }
        };
        eh.useErr(err);
        eh.useOut(out);

        List<CommandLine> commandLines;
        try {
            commandLines = testCommand.parse(args);
        } catch (ParameterException pe) {
            eh.handleException(pe, err, eh.ansi(), args);
            return exitCode.get();
        }

        Optional<CommandLine> helpCmdOptional = commandLines.stream()
                .filter(CommandLine::isUsageHelpRequested).findFirst();
        if (helpCmdOptional.isPresent()) {
            CommandLine commandLine = helpCmdOptional.get();
            showUsage(out, commandLine);
            return 0;
        }

        Optional<CommandLine> helpOptional = commandLines.stream()
                .filter(c -> c.getCommandSpec().helpCommand())
                .findFirst();
        if (helpOptional.isPresent()) {
            showUsage(out, testCommand);
            return 0;
        }

        CommandLine commandLine = new CommandLine(new CommandLineInterpreter());

        commandLine.parseWithHandlers(new RunLast(), eh, args);
        return exitCode.get();
    }

    private static void showUsage(PrintStream printStream, CommandLine commandLine) {
        Help help = new Help(commandLine.getCommandSpec(), defaultColorScheme(AUTO));

        int width = Stream.of(System.getProperty("COLUMNS"), System.getenv("COLUMNS"))
                .map(StringUtils::trimToEmpty)
                .filter(StringUtils::isNotEmpty)
                .filter(s -> s.matches("\\d{2,3}"))
                .mapToInt(Integer::parseInt)
                .findFirst()
                .orElse(-1);

        if (width >= 80 && width < 400) {
            commandLine.setUsageHelpWidth(width);
        }

        StringBuilder usage = CommandLine.usage(new StringBuilder(), help);
        printStream.println(usage);
    }
}
