package com.myapp.videotools.commandline;

import com.myapp.videotools.misc.Broadcast;
import com.myapp.videotools.misc.SeleniumHolder;
import org.apache.commons.lang3.StringUtils;
import picocli.CommandLine.Mixin;
import picocli.CommandLine.Option;
import picocli.custom.CustomTypeConverters.ReadableFile;
import picocli.custom.CustomTypeConverters.WriteableFile;

import java.io.File;
import java.io.IOException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import static com.myapp.videotools.commandline.CheckForDownloadableMoviesCommand.CMD_NAME;
import static picocli.CommandLine.Command;


/**
 * Queries the ORF TvThek and checks if there are movies
 * to download by trying to find matches against IMDB.
 */
@Command(name = CMD_NAME,
        aliases = {"--check-tvthek", "-tv"},
        description = "Create a single frame snapshot image of a videofile")
public class CheckForDownloadableMoviesCommand extends VideoToolsBaseCommand {

    @SuppressWarnings("WeakerAccess")
    public static final String CMD_NAME = "tvthek";
    @SuppressWarnings("WeakerAccess")
    public static final String OPT_THRESHOLD = "--duration-threshold";
    @SuppressWarnings("WeakerAccess")
    public static final String OPT_WRITE_CACHE_FILE = "--write-cache-file";
    @SuppressWarnings("WeakerAccess")
    public static final String OPT_READ_CACHE_FILE = "--read-cache-file";

    private static final String LINE_SEPARATOR = System.getProperty("line.separator");
    public static final DateTimeFormatter FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm");

    private SeleniumHolder seleniumHolder = new SeleniumHolder();

    @Mixin
    private Filters filterSettings = new Filters();

    @Option(names = {OPT_READ_CACHE_FILE, "-rcf"},
            converter = ReadableFile.class,
            description = "Read broadcasts from a file instead of fetching from ORF.")
    private File broadcastInput = null;

    @Option(names = {OPT_WRITE_CACHE_FILE, "-wcf"},
            converter = WriteableFile.class,
            description = "Writes fetched broadcasts to a file.")
    private File broadcastOutput = null;

    @Override
    protected void executeImpl() {
        if (broadcastInput != null && broadcastOutput != null) {
            throw paramEx(OPT_WRITE_CACHE_FILE, "");
        }

        List<Broadcast> allBroadcasts = collectAllBroadcasts();

        List<Broadcast> interesting = allBroadcasts.stream()
                .filter(bc -> filterSettings.isInteresting(bc, allBroadcasts))
                .sorted(Comparator.comparing(bc -> bc.getTitle().replaceFirst("^AD \\| ", "")))
                .collect(Collectors.toList());

        System.out.println(printMatchesInTextTable(allBroadcasts, interesting));
        System.out.println();
        System.out.println(filterSettings.showFilterStatistics());

        if (filterSettings.isShowUnNecessaryFilters()) {
            filterSettings.showUnnecessaryFilters();
        }
    }

    private List<Broadcast> collectAllBroadcasts() {
        List<Broadcast> allBroadcasts;

        if (broadcastInput == null) {
            allBroadcasts = seleniumHolder.fetchAllBroadcasts();
            if (broadcastOutput != null) {
                try {
                    Broadcast.toFile(broadcastOutput, allBroadcasts);
                } catch (IOException e) {
                    throw execEx("Unable to write to " + broadcastInput, e);
                }
            }
        } else {
            try {
                allBroadcasts = Broadcast.fromFile(broadcastInput);
            } catch (IOException e) {
                throw execEx("Unable to read from " + broadcastInput, e);
            }
        }

        return allBroadcasts;
    }


    private String printMatchesInTextTable(List<Broadcast> allBroadcasts, List<Broadcast> interesting) {
        StringBuilder sb = new StringBuilder();

        // collect string lengths for a nice table

        AtomicInteger longestTitleLen = new AtomicInteger(0);
        AtomicInteger longestAvailabilityLen = new AtomicInteger(0);
        AtomicInteger longestChannelLen = new AtomicInteger(0);
        AtomicInteger longestDurationLen = new AtomicInteger(0);
        AtomicInteger longestDescriptionLen = new AtomicInteger(0);

        interesting.parallelStream().forEach(bc -> {
            int titleLen = bc.getTitle().length();
            if (longestTitleLen.get() < titleLen) {
                longestTitleLen.set(titleLen);
            }

            int channelLen = bc.getTvChannel().length();
            if (longestChannelLen.get() < channelLen) {
                longestChannelLen.set(channelLen);
            }

            int availLen = bc.getAvailability().length();
            if (longestAvailabilityLen.get() < availLen) {
                longestAvailabilityLen.set(availLen);
            }

            int durationLen = bc.getDuration().length();
            if (longestDurationLen.get() < durationLen) {
                longestDurationLen.set(durationLen);
            }

            int descriptionLen = StringUtils.length(bc.getDescription());
            if (longestDescriptionLen.get() < descriptionLen) {
                longestDescriptionLen.set(descriptionLen);
            }
        });

        // render the table

        interesting.forEach(bc -> {
            sb.append(bc.getTitle());
            for (int i = longestTitleLen.get() - bc.getTitle().length(); i > 0; i--) { sb.append(" "); }
            sb.append(" | ");
            sb.append(bc.getTvChannel());
            for (int i = longestChannelLen.get() - bc.getTvChannel().length(); i > 0; i--) { sb.append(" "); }
            sb.append(" | ");
            sb.append(bc.getDuration());
            for (int i = longestDurationLen.get() - bc.getDuration().length(); i > 0; i--) { sb.append(" "); }
            sb.append(" | noch ");
            sb.append(bc.getAvailability());
            for (int i = longestAvailabilityLen.get() - bc.getAvailability().length(); i > 0; i--) { sb.append(" "); }
            sb.append(" | ");
            sb.append(bc.getDescription());
            for (int i = longestDescriptionLen.get() - bc.getDescription().length(); i > 0; i--) { sb.append(" "); }
            sb.append(" | ");
            sb.append(bc.getStreamUrl());
            sb.append(LINE_SEPARATOR);
        });

        // add a bottom line

        sb.append(LINE_SEPARATOR);
        sb.append("Items total: ").append(allBroadcasts.size());
        sb.append(", kept: ").append(interesting.size());
        sb.append(", filtered: ").append(allBroadcasts.size() - interesting.size());
        sb.append(" Fetched at: ").append(DateTimeFormatter.ISO_DATE_TIME.format(LocalDateTime.now()));

        return sb.toString();
    }

    public static void main(String[] args) {
        CheckForDownloadableMoviesCommand cmd = new CheckForDownloadableMoviesCommand();
//        cmd.filterSettings.setShowUnNecessaryFilters(true);
        cmd.executeImpl();
    }
}
