package com.myapp.videotools.commandline;

import com.myapp.videotools.IPathCalculator;
import com.myapp.videotools.IVideoFileParser;
import com.myapp.videotools.IVideoThumbnailer;
import com.myapp.videotools.VideoFile;
import com.myapp.videotools.impl.Application;
import com.myapp.videotools.impl.FileHierarchyCopying;
import com.myapp.videotools.impl.NextToSourceFile;
import com.myapp.videotools.misc.AppStatistics;
import com.myapp.videotools.misc.StatisticsShutDownHook;
import com.myapp.videotools.misc.Util;
import org.assertj.core.util.VisibleForTesting;
import picocli.CommandLine.Mixin;
import picocli.custom.CustomTypeConverters.ExistingWriteableDirectory;
import picocli.custom.CustomTypeConverters.PositiveInteger;
import picocli.custom.CustomTypeConverters.ReadableDirectory;
import picocli.custom.CustomTypeConverters.WriteableFile;
import picocli.custom.MessageOnlyException;

import java.io.File;
import java.io.IOException;

import static com.myapp.videotools.IVideoThumbnailer.*;
import static com.myapp.videotools.IVideoThumbnailer.OverwritePolicy.OVERWRITE;
import static com.myapp.videotools.IVideoThumbnailer.OverwritePolicy.SKIP;
import static com.myapp.videotools.commandline.BigPictureCommand.CMD_BIGPIC;
import static com.myapp.videotools.commandline.InOut.*;
import static com.myapp.videotools.misc.Util.isSameFile;
import static picocli.CommandLine.Command;
import static picocli.CommandLine.Help.Visibility.ALWAYS;
import static picocli.CommandLine.Option;

/**
 * Creates a big picture for a video file
 */
@Command(name = CMD_BIGPIC,
        aliases = {"--create-big-picture", "-bigpic"},
        description = "Combines a set of multiple thumbnails of a video file to " +
                "a big-picture image file.")
public class BigPictureCommand extends VideoToolsBaseCommand {

    static final String CMD_BIGPIC = "bigpic";
    static final String OPT_RECURSIVE =  "--recursive";
    static final String OPT_INPUT_DIR = "--video-root-dir";
    static final String OPT_OUTPUT_DIR = "--big-picture-root-dir";
    static final String OPT_ROWS = "--rows";
    static final String OPT_COLUMNS = "--columns";
    static final String OPT_BP_PREFIX = "--big-picture-prefix";
    static final String OPT_BP_SUFFIX = "--big-picture-suffix";
    static final String OPT_SUPPRESS_STATISTICS = "--suppress-statistics";
    static final String OPT_ANIMATED = "--animated-gif";
    static final String OPT_RANGE_SELECTION = "--range-selection";

    @Option(names = OPT_RANGE_SELECTION,
            description = "Which part of the video file will be considered for the thumbnails.",
            converter = RangeSelection.Converter.class)
    private RangeSelection partSelection;

    @Mixin
    private FrameDimensions dims = new FrameDimensions();

    @Option(names = {OPT_RECURSIVE, "-R"},
            description = "Recurse into directories and create big-pics for every video file found.")
    boolean recursive;

    @Option(names = {OPT_INPUT_DIR, "-vr"},
            converter = ReadableDirectory.class,
            paramLabel = "DIR",
            description = "Root directory where the videos are located")
    File inDir;

    @Option(names = {OPT_OUTPUT_DIR, "-bpr"},
            converter = ExistingWriteableDirectory.class,
            paramLabel = "DIR",
            description = "directory where the big-picture files will be stored. " +
                    "if not specified, it will be written next to the video file.")
    File outDir;

    @Option(names = {OPT_ROWS, "-r"},
            converter = PositiveInteger.class,
            defaultValue = "" + DEFAULT_BIG_PIC_ROWS,
            showDefaultValue = ALWAYS,
            paramLabel = "INT",
            description = "Number of rows in the big-picture grid.")
    int rows;

    @Option(names = {OPT_COLUMNS, "-c"},
            converter = PositiveInteger.class,
            defaultValue = "" + DEFAULT_BIG_PIC_COLS,
            showDefaultValue = ALWAYS,
            paramLabel = "INT",
            description = "Number of columns in the big-picture grid.")
    int cols;

    @Option(names = {OPT_BP_PREFIX, "-bp"},
            defaultValue = DEFAULT_BIGPIC_FILENAME_PREFIX,
            description = "Prefix of the big-picture files")
    String bigPicPrefix;

    @Option(names = {OPT_BP_SUFFIX, "-bs"},
            defaultValue = DEFAULT_BIGPIC_FILENAME_SUFFIX,
            showDefaultValue = ALWAYS,
            description = "Suffix of the big-picture files")
    String bigPicSuffix;

    @Option(names = {OPT_SUPPRESS_STATISTICS},
            defaultValue = "false",
            hidden = true,
            description = "Dont show stats after generating recursively")
    boolean suppressStatistics;

    @Option(names = {OPT_ANIMATED, "-gif"},
            defaultValue = "false",
            description = "Generate animated gifs as big-picture")
    boolean animated;

    @VisibleForTesting
    static StatisticsShutDownHook statisticsShutDownHook = new StatisticsShutDownHook();

    @Override
    protected void executeImpl() {
        if (bigPicPrefix == null || bigPicPrefix.contains("%")) {
            throw paramEx(OPT_BP_PREFIX, "must not contain '%' symbol");
        }

        if (recursive) {
            executeRecursive();
        } else {
            executeSingle();
        }
    }

    private void executeSingle() {
        String suffix = " when option " + OPT_RECURSIVE + " is enabled.";
        String notWithOutputFile = "is unsupported with explicit " + OPT_OUTPUT_FILE + " option.";

        if (inDir != null)
            throw paramEx(OPT_INPUT_DIR, "is only applicable" + suffix);
        // this will not detect when the user overwrites the default, but does it matter?
        if (io.outFile != null && !DEFAULT_BIGPIC_FILENAME_PREFIX.equals(bigPicPrefix))
            throw paramEx(OPT_BP_PREFIX, notWithOutputFile);
        // this will not detect when the user overwrites the default, but does it matter?
        if (io.outFile != null && !DEFAULT_BIGPIC_FILENAME_SUFFIX.equals(bigPicSuffix)) {
            throw paramEx(OPT_BP_SUFFIX, notWithOutputFile);
        }
        if (outDir != null)
            throw paramEx(OPT_OUTPUT_DIR, "is only applicable" + suffix);

        if (io.inFile == null) {
            throw paramEx(OPT_INPUT_FILE, "Must be specified.");
        } else if (io.outFile != null && io.outFile.exists() && isSameFile(io.inFile, io.outFile)) {
            throw paramEx(OPT_INPUT_FILE, "refers to the same file as option " + OPT_OUTPUT_FILE);
        }

        boolean impliedTarget = false;
        IVideoThumbnailer tn = instantiateNailer();
        if (io.outFile == null) {
            NextToSourceFile pathCalc = new NextToSourceFile();
            pathCalc.setSuffix(bigPicSuffix);
            pathCalc.setPrefix(bigPicPrefix);
            tn.setPathCalculator(pathCalc);

            String implicitTarget = pathCalc.getTargetPath(io.inFile);
            try {
                io.outFile = WriteableFile.getWriteableFile(implicitTarget);
                impliedTarget = true;

            } catch (MessageOnlyException e) {
                throw paramEx(OPT_OUTPUT_FILE, "Implicit outputfile " + e + ": " + implicitTarget);
            }
        }
        if (!io.overWriteExisting && io.outFile.exists()) {
            throw paramEx(OPT_OVERWRITE_EXISTING, "is not enabled but target file exists.");
        }

        log.debug("  Configuring the video thumbnailer instance ...");

        try {
            VideoFile videoFile = new VideoFile(io.inFile);
            IVideoFileParser parser = Application.getInstance().createVideoFileParser();
            videoFile.parse(parser);
            tn.setVideoFile(videoFile);

            if (impliedTarget) {
                log.info("  Output unspecified - writing big-picture next to the video file.");
            }
            tn.createBigPicture(io.outFile);

        } catch (IOException e) {
            throw execEx("could not create big-picture for: "
                    + io.inFile.getName() + " (" + e.getMessage() + ")", e);
        }
    }

    private void executeRecursive() {
        String suffix = " when option " + OPT_RECURSIVE + " is enabled.";
        if (inDir == null) throw paramEx(OPT_INPUT_DIR, "must be specified" + suffix);
        if (io.inFile != null) throw paramEx(OPT_INPUT_FILE, "cannot be used" + suffix);
        if (io.outFile != null) throw paramEx(OPT_OUTPUT_FILE, "cannot be used" + suffix);

        log.info("Creating big-pictures recursively ...");
        log.debug("  Configuring the video thumbnailer instance ...");
        IVideoThumbnailer tn = instantiateNailer();
        tn.setVideoRootDir(inDir);

        IPathCalculator pathCalculator;
        if (outDir == null) {
            log.info("  Files will be written next to the source file");
            pathCalculator = new NextToSourceFile();
        } else {
            log.info("  Files will be written into:             {}", outDir);
            pathCalculator = new FileHierarchyCopying(inDir, outDir);
        }
        pathCalculator.setPrefix(bigPicPrefix);
        String suf = bigPicSuffix;
        if (animated && bigPicSuffix.equals(DEFAULT_BIGPIC_FILENAME_SUFFIX)) {
            suf = DEFAULT_BIGPIC_FILENAME_SUFFIX_ANIMATED;
        }
        pathCalculator.setSuffix(suf);
        tn.setPathCalculator(pathCalculator);
        tn.setAnimated(animated);

        log.info("  Searching for videos in:                {} ...", tn.getVideoRootDir());
        if (!suppressStatistics) {
            AppStatistics instance = AppStatistics.getInstance();
            if (instance.getApplicationStart() < 0) {
                instance.setApplicationStart();
            }
            Runtime.getRuntime().addShutdownHook(statisticsShutDownHook);
        }

        tn.createBigPictureRecursively();

        log.info("Created big-pictures recursively.");
    }

    protected IVideoThumbnailer instantiateNailer() {
        Application application = Application.getInstance();

        if (log.isDebugEnabled()) {
            log.debug(Util.getOsInfoString());
        }

        IVideoThumbnailer thumbnailer = application.createVideoThumbnailer();

        thumbnailer.setPreferredWidth(dims.thumbnailWidth);
        thumbnailer.setPreferredHeight(dims.thumbnailHeight);
        thumbnailer.setBigPictureRows(rows);
        thumbnailer.setBigPictureCols(cols);
        thumbnailer.setOverwritePolicy(io.overWriteExisting ? OVERWRITE : SKIP);
        thumbnailer.setRangeSelection(partSelection);

        return thumbnailer;
    }
}
