package com.myapp.videotools;

import com.myapp.videotools.misc.AppStatistics;
import org.assertj.core.util.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;

import static com.myapp.util.format.TimeFormatUtil.formatTimeTo2Digits;


/**
 * data class to back the information of following ffmpeg call:
 * 
 * <pre>
 * ffmpeg -i /data/stuff/vidfile.flv 2&gt;&amp;1 | grep -E "Input|Duration|Stream"
 * Input #0, flv, from '/data/stuff/vidfile.flv':
 *   Duration: 00:03:50.2, start: 0.000000, bitrate: 32 kb/s
 *   Stream #0.0: Video: vp6f, yuv420p, 640x480, 30.00 fps(r)
 *   Stream #0.1: Audio: mp3, 22050 Hz, stereo, 32 kb/s
 * </pre>
 */
@SuppressWarnings({"WeakerAccess", "UnusedReturnValue"})
public final class VideoFile {

    private static final Logger log = LoggerFactory.getLogger(VideoFile.class);
    
    private final File file;
    private boolean parsed = false;
    
    private String videoFormat = null;
    private String videoCodec = null;
    
    private String audioCodec = null;
    private String audioChannelType = null;
    
    private int audioSampleRate = -1;
    private int audioBytesPerSecond = -1;
    private int totalBytesPerSecond = -1;

    private int videoBytesPerSecond = -1;
    private double videoFramesPerSecond = -1;

    private int videoWidth = -1;
    private int videoHeight = -1;
    
    private long lengthMillis = -1;
    private long videoStartOffsetMillis = 0;
    
    private AppStatistics statistics = AppStatistics.getInstance();

    
    
    public VideoFile(File file) {
        if (file == null) {
            throw new NullPointerException();
        }
        this.file = file;
    }
    
    public VideoFile(String path) {
        this(new File(path));
    }
    
    
    
    
    public void parse(IVideoFileParser parser) throws IOException {
        final long start = System.currentTimeMillis();
        try {
            parse0(parser);
            parsed = true;
            statistics.incrementFilesParsed();
            
        } catch (IOException | RuntimeException e) {
            statistics.incrementParseFails();
            throw e;
            
        } finally {
            long timeNeeded = System.currentTimeMillis() - start;
            statistics.addTimeSpentWithParsingMetadata(timeNeeded);
        }
    }
    
    private void parse0(IVideoFileParser parser) throws IOException {         
        if (parser == null)      throw new NullPointerException();
        if ( ! file.exists())    throw new FileNotFoundException(file.getAbsolutePath());
        if (file.isDirectory())  throw new IllegalArgumentException(file.getAbsolutePath());
        
        parser.parse(this);
        String path = file.getPath();

        if (videoFormat == null) {
            log.trace("      videoFormat could not be determined            : {}", path);
        }
        if (videoCodec == null) {
            log.trace("      videoCodec could not be determined          : {}", path);
        }
        if (audioCodec == null) {
            log.trace("      audioCodec could not be determined          : {}", path);
        }
        if (audioChannelType == null) {
            log.trace("      audioChannelType could not be determined    : {}", path);
        }
        if (audioSampleRate <= 0) {
            log.trace("      audioSampleRate could not be determined     : {}", path);
        }
        if (totalBytesPerSecond <= 0) {
            log.trace("      bytesPerSecond could not be determined      : {}", path);
        }
        if (videoFramesPerSecond <= 0) {
            log.trace("      framesPerSecond could not be determined     : {}", path);
        }
        if (videoWidth <= 0) {
            log.warn("      width could not be determined               : {}", path);
        }
        if (videoHeight <= 0) {
            log.warn("      height could not be determined              : {}", path);
        }
        if (audioBytesPerSecond <= 0) {
            log.trace("      audioBytesPerSecond could not be determined : {}", path);
        }

        if (lengthMillis == -1) {
            throw new RuntimeException("Unable to parse video file: " + path);
        }
    }

    public boolean isParsed() {
        return parsed;
    }
    
    public File getFile() {
        return file;
    }

    public String getAudioChannelType() {
        return audioChannelType;
    }

    public String getAudioCodec() {
        return audioCodec;
    }

    public int getAudioSampleRate() {
        return audioSampleRate;
    }

    public int getTotalBytesPerSecond() {
        return totalBytesPerSecond;
    }

    public String getVideoFormat() {
        return videoFormat;
    }

    public double getVideoFramesPerSecond() {
        return videoFramesPerSecond;
    }

    public int getVideoHeight() {
        return videoHeight;
    }

    public long getLengthMillis() {
        return lengthMillis;
    }

    public String getVideoCodec() {
        return videoCodec;
    }

    public int getVideoWidth() {
        return videoWidth;
    }

    public double getLengthSeconds() {
        return ((double) lengthMillis) / 1000.0d;
    }
    
    public String getFormattedLength() {
        return formatTimeTo2Digits(getLengthSeconds());
    }
    
    public String getName() {
        return file.getName();
    }

    public VideoFile setVideoFormat(String fileType) {
        this.videoFormat = fileType;
        return this;
    }

    public VideoFile setVideoCodec(String videoCodec) {
        this.videoCodec = videoCodec;
        return this;
    }

    public VideoFile setAudioCodec(String audioCodec) {
        this.audioCodec = audioCodec;
        return this;
    }

    public VideoFile setAudioChannelType(String type) {
        this.audioChannelType = type;
        return this;
    }

    public VideoFile setAudioSampleRate(int rate) {
        this.audioSampleRate = rate;
        return this;
    }

    public VideoFile setTotalBytesPerSecond(int bps) {
        this.totalBytesPerSecond = bps;
        return this;
    }

    public VideoFile setVideoFramesPerSecond(double fps) {
        this.videoFramesPerSecond = fps;
        return this;
    }

    public VideoFile setVideoWidth(int width) {
        this.videoWidth = width;
        return this;
    }

    public VideoFile setVideoHeight(int height) {
        this.videoHeight = height;
        return this;
    }

    public VideoFile setLengthMillis(long lengthMillis) {
        this.lengthMillis = lengthMillis;
        return this;
    }

    public VideoFile setVideoStartOffsetMillis(long millis) {
        this.videoStartOffsetMillis = millis;
        return this;
    }

    public long getVideoStartOffsetMillis() {
        return videoStartOffsetMillis;
    }

    public VideoFile setVideoBytesPerSecond(int bps) {
        this.videoBytesPerSecond = bps;
        return this;
    }

    public int getVideoBytesPerSecond() {
        return videoBytesPerSecond;
    }

    public VideoFile setAudioBytesPerSecond(int bps) {
        this.audioBytesPerSecond = bps;
        return this;
    }

    public int getAudioBytesPerSecond() {
        return audioBytesPerSecond;
    }
    @Override
    public String toString() {
        StringBuilder s = new StringBuilder();
        if (file != null) {
            s.append(getName());
        }
        if (videoFormat != null) {
            s.append(" type:");
            s.append(getVideoFormat());
        }
        if (lengthMillis > 0) {
            s.append(", duration: ");
            s.append(getFormattedLength());
        }
        if (totalBytesPerSecond > 0) {
            s.append(", bitrate:");
            s.append(getTotalBytesPerSecond());
            s.append("b/s");
        }
        if (s.length() > 0) {
            s.append(", ");
        }
        s.append("VIDEO:[");
        if (videoCodec != null) {
            s.append("codec=");
            s.append(getVideoCodec());
            s.append(" ");
        }
        if (videoFramesPerSecond > 0) {
            s.append("frames=");
            s.append(getVideoFramesPerSecond());
            s.append("fps ");
        }
        if (videoWidth > 0 && videoHeight > 0) {
            s.append("dimension=");
            s.append(getVideoWidth());
            s.append("x");
            s.append(getVideoHeight());
            s.append(" ");
        }
        if (videoBytesPerSecond > 0) {
            s.append("bitrate=");
            s.append(getVideoBytesPerSecond());
            s.append("b/s ");
        }
        if (videoStartOffsetMillis > 0) {
            s.append("offset=");
            s.append(getVideoStartOffsetMillis());
            s.append("ms");
        }
        s.append("]");
        s.append(", ");
        s.append("AUDIO:[");
        if (audioCodec != null) {
            s.append("codec=");
            s.append(getAudioCodec());
            s.append(" ");
        }
        if (audioSampleRate > 0) {
            s.append("samplerate=");
            s.append(getAudioSampleRate());
            s.append(" ");
        }
        if (audioChannelType != null) {
            s.append("channeltype=");
            s.append(getAudioChannelType());
            s.append(" ");
        }
        if (audioBytesPerSecond > 0) {
            s.append("bitrate=");
            s.append(getAudioBytesPerSecond());
            s.append("b/s");
        }
        s.append("]");
        return s.toString();
    }

    @VisibleForTesting
    public VideoFile _setParsed() {
        parsed = true;
        return this;
    }
}
