package com.myapp.videotools;

import com.myapp.videotools.commandline.RangeSelection;
import com.myapp.videotools.commandline.RangeSelection.Coords;
import com.myapp.videotools.impl.Application;
import com.myapp.videotools.impl.NextToSourceFile;
import com.myapp.videotools.misc.AppStatistics;
import com.myapp.videotools.misc.Util.DirsFirstAlphabeticFileComparator;
import org.apache.commons.io.FileUtils;
import org.assertj.core.api.Assertions;
import org.assertj.core.util.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import static org.assertj.core.api.Assertions.assertThat;


public abstract class AbstractVideoThumbnailer implements IVideoThumbnailer {

    private static final Logger LOG = LoggerFactory.getLogger(AbstractVideoThumbnailer.class);
    private static final String TMP_DIR = System.getProperty("java.io.tmpdir");
    private static final DirsFirstAlphabeticFileComparator FILE_CMP = new DirsFirstAlphabeticFileComparator();

    private FileFilter videoFileFilter = new DefaultVideoFileFilter();
    @VisibleForTesting @SuppressWarnings("WeakerAccess")
    protected IVideoFileParser videoFileParser = Application.getInstance().createVideoFileParser();
    private IPathCalculator targetPathGenerator = new NextToSourceFile();
    protected IImageMerger merger = Application.getInstance().createImageMerger();
    protected AppStatistics statistics = AppStatistics.getInstance();

    private int preferredWidth = DEFAULT_THUMB_WIDTH;
    private int preferredHeight = DEFAULT_THUMB_HEIGHT;
    private int bigPictureRows = DEFAULT_BIG_PIC_ROWS;
    private int bigPictureCols = DEFAULT_BIG_PIC_COLS;

    protected VideoFile videoFile;
    private File videoRootDir = null;
    private boolean animated = false;

    private OverwritePolicy overwritePolicy = OverwritePolicy.SKIP;
    private RangeSelection rangeSelection;


    protected AbstractVideoThumbnailer() {
    }

    @Override
    public void createBigPictureRecursively() {
        createBigPictureRecursively0(getVideoRootDir());
    }

    /**
     * decides if a file is a directory or a regular file and invokes the
     * designated method for creating big-pictures recursively.
     * 
     * @param file
     *          the file or directory
     */
    private void createBigPictureRecursively0(File file) {
        if (file.isDirectory()) {
            createBigPictureRecursivelyForDirectory(file);
        } else if (file.isFile()) {
            createBigPictureRecursivelyForFile(file);
        } else {
            throw new RuntimeException("File is neither a file nor a directory: " + file);
        }
    }

    /**
     * crawl into a directory recursively and invoke pig-picture process for all
     * children. during recursively creating thumbnails, files of type directory
     * will be handled by this method.
     * 
     * @param directory
     *            the directory to crawl into
     */
    private void createBigPictureRecursivelyForDirectory(File directory) {
        LOG.info("  Entering directory:                     {}", directory);

        File[] children = directory.listFiles();
        assertThat(children).isNotNull();
        Arrays.sort(children, FILE_CMP);
        Arrays.stream(children).forEach(this::createBigPictureRecursively0);

        LOG.info("  Exiting directory:                      {}", directory);
        if (!directory.equals(videoRootDir)) {
            LOG.info("  Now in directory:                       {}", directory.getParent());
        }
    }

    /**
     * the file will be parsed and a big-picture will be created from it.<br>
     * then the thumbnails will be created in a temp dir.<br>
     * the thumbnails will be merged to a big-picture and then deleted.<br>
     * if the file argument
     * 
     * <ul>
     * <li>is not accepted by the current videoFileFilter</li>
     * <li>already has a thumbnail picture</li>
     * <li>an error occured while parsing the video metadata for this file</li>
     * </ul>
     * 
     * this method will do nothing.
     * 
     * @param file
     *            the file to create the big pic for.
     */
    private void createBigPictureRecursivelyForFile(File file) {
        String path = file.getPath();
        StringBuilder bui = new StringBuilder();
        bui.append("    Creating big-picture for file:        ").append(path).append(" ... ");
        for (int i = path.length(); i < 60; i++) {
            bui.append(' ');
        }
        
        if ( ! videoFileFilter.accept(file)) {
            LOG.info(bui.append("        SKIP. (filtered)").toString());
            statistics.incrementSkippedBecauseFiltered();
            return;
        }
        
        VideoFile v = new VideoFile(file);
        setVideoFile(v);

        File vid = videoFile.getFile();
        File bigPicFile = new File(targetPathGenerator.getTargetPath(vid));
        if (bigPicFile.exists()) {
            OverwritePolicy policy = getOverwritePolicy();
            switch (policy) {
                case SKIP: {
                    LOG.info(bui.append("        SKIP. (already has big pic)").toString());
                    statistics.incrementSkippedBecauseExistingTarget();
                    return;
                }
                case OVERWRITE: {
                    if (bigPicFile.isFile() && bigPicFile.canWrite()) {
                        bui.append("        OVERWRITING existing file!");
                        // overwrite statistics could be implemented here
                        break;
                    } else {
                        LOG.warn(bui.append("        SKIP. (file not writeable)").toString());
                        statistics.incrementSkippedBecauseExistingTarget();
                        return;
                    }
                }
            }
        }

        File par = bigPicFile.getParentFile();
        assertThat(par.isDirectory() || par.mkdirs()).as("not a directory " + par).isTrue();
        LOG.info(bui.toString());

        boolean yetParsed = parseIfNotYetParsed();
        if (!yetParsed) {
            return;
        }

        try {
            boolean ok;
            if (animated) {
                ok = createAnimatedBigPicture(bigPicFile, 10); // TODO: parameterize
            } else {
                ok = createBigPicture(bigPicFile);
            }
            if ( ! ok) {
                LOG.error("    FAIL, unable to create big-pic for: {}", file.getName());
                return;
            }
    
        } catch (Exception e) {
            LOG.error("      Could not create big-picture for: {}", file.getName());
            LOG.trace("      Stacktrace:", e);
            LOG.error("    FAIL, an error ({}) occured while create big-pic for: {}", e, file.getName());
            return;
        }

        LOG.info("    OK, created big-picture for:          {}", file.getName());
    }

    protected boolean parseIfNotYetParsed() {
        if (videoFile.isParsed()) {
            return true;
        }

        try {
            videoFile.parse(videoFileParser);
            return videoFile.isParsed();

        } catch (Exception e) {
            File file = videoFile.getFile();
            LOG.error("      Could not determine video-metadata for: {}", file.getName());
            LOG.trace("      Stacktrace:", e);
            LOG.error("    FAIL, an error ({}) occured while parsing {}", e, file.getName());
            return false;
        }
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append(getClass().getSimpleName()).append(" [file=");
        
        if (videoFile != null && videoFile.getFile() != null)
            builder.append(videoFile.getFile().getName());
        else 
            builder.append("<NULL>");
        
        builder.append(", bigPic r*c=");
        builder.append(bigPictureRows);
        builder.append("*");
        builder.append(bigPictureCols);
        builder.append(", thumbs: h*w=");
        builder.append(preferredHeight);
        builder.append("*");
        builder.append(preferredWidth);
        if (videoRootDir != null) {
            builder.append(", videoRootDir=");
            builder.append(videoRootDir.getName());
        }
        builder.append("]");
        return builder.toString();
    }
    
    /**
     * calculates the path to a temporary dir where thumbnails of the current
     * videofiles will be created before merging to a big-picture from them.
     * 
     * @return a directory where thumbnails will be cached
     */
    public File calculateThumbnailTempDir() {
        return new File(TMP_DIR, "thumbnails-tempdir-" + videoFile.getFile().getName());
    }
    /**
     * this will create the temporary targetdir, if not already existing. if
     * existing, all data will be deleted in the directory.
     * 
     * @param targetDir
     *            the directory used to temporarily cache the thumbnails.
     */
    public void createOrWipeTempThumbnailDir(File targetDir) {
        if ( ! targetDir.exists()) {
            assertThat(targetDir.mkdirs()).isTrue();
            
        } else {
            File[] subfiles = targetDir.listFiles();
            assertThat(subfiles).isNotNull();
            
            if (subfiles.length > 0) {
                LOG.warn("          TARGETDIR NOT EMPTY! sweeping {} files from: {}", subfiles.length, targetDir);
                Arrays.stream(subfiles)
                        .forEach(f -> {
                            FileUtils.deleteQuietly(f);
                            assertThat(f.exists()).as("Could not delete " + f).isFalse();
                        });
            }
        }      
    }

    @Override
    public VideoFile getVideoFile() {
        return videoFile;
    }

    @Override
    public File getVideoRootDir() {
        return videoRootDir;
    }

    @Override
    public void setVideoRootDir(File dir) {
        videoRootDir = dir;
    }

    @Override
    public void setVideoFile(VideoFile vf) {
        videoFile = vf;
    }

    @Override
    public int getPreferredWidth() {
        return preferredWidth;
    }

    @Override
    public int getPreferredHeight() {
        return preferredHeight;
    }

    @Override
    public void setPreferredWidth(int w) {
        LOG.debug("  picture width: '{}'", w);
        preferredWidth = w;
    }

    @Override
    public void setPreferredHeight(int h) {
        LOG.debug("  picture height: '{}'", h);
        preferredHeight = h;
    }

    @Override
    public void setPathCalculator(IPathCalculator pc) {
        LOG.debug("  big-picture path algorithm: '{}'", pc.getClass());
        targetPathGenerator = pc;
    }

    @Override
    public IPathCalculator getPathCalculator() {
        return targetPathGenerator;
    }

    @Override
    public int getBigPictureCols() {
        return bigPictureCols;
    }

    @Override
    public int getBigPictureRows() {
        return bigPictureRows;
    }

    @Override
    public void setBigPictureCols(int c) {
        LOG.debug("  big-picture columns: '{}'", c);
        bigPictureCols = c;
    }

    @Override
    public void setBigPictureRows(int r) {
        LOG.debug("  big-picture rows: '{}'", r);
        bigPictureRows = r;
    }

    @Override
    public FileFilter getFileFilter() {
        return videoFileFilter;
    }

    @Override
    public void setFileFilter(FileFilter e) {
        videoFileFilter = e;
    }

    @Override
    public void setOverwritePolicy(OverwritePolicy overwritePolicy) {
        this.overwritePolicy = overwritePolicy;
    }

    @Override
    public OverwritePolicy getOverwritePolicy() {
        return overwritePolicy;
    }

    @Override
    public boolean isAnimated() {
        return animated;
    }

    @Override
    public void setAnimated(boolean animated) {
        this.animated = animated;
    }

    @Override
    public void setRangeSelection(RangeSelection rangeSelection) {
        this.rangeSelection = rangeSelection;
    }

    public static long calculateOffsetMillis(RangeSelection selection, VideoFile file, int sliceIndex, int sliceCount) {
        Assertions.assertThat(sliceIndex).isGreaterThanOrEqualTo(0);
        Assertions.assertThat(sliceCount).isGreaterThan(0);

        RangeSelection rs = selection != null ? selection : RangeSelection.COMPLETELY;
        Coords coords = rs.calculate(file);

        long selectionLength = coords.getLengthMillis();

        Assertions.assertThat(selectionLength).isGreaterThan(0L);
        long offsetWithinSelection = Math.round(((double)selectionLength / sliceCount) * sliceIndex);
        long selectionOffset = coords.getOffsetMillis();
        return selectionOffset + offsetWithinSelection;
    }

    public long calculateOffsetMillis(int sliceIndex, int numberOfSlices) {
        return calculateOffsetMillis(rangeSelection, videoFile, sliceIndex, numberOfSlices);
    }

    @SuppressWarnings("WeakerAccess")
    public RangeSelection getRangeSelection() {
        return rangeSelection;
    }

    @VisibleForTesting
    public Process __startProcess(ProcessBuilder pb) throws IOException {
        return pb.start();
    }

    @VisibleForTesting
    public int __waitForProcess(Process p, List<String> command) throws InterruptedException {
        return p.waitFor();
    }

    @VisibleForTesting
    public void __awaitTermination(ExecutorService service, long maxWaitMillis, String description)
            throws InterruptedException {
        service.shutdown();
        service.awaitTermination(maxWaitMillis, TimeUnit.MILLISECONDS);
        Assertions.assertThat(service.isTerminated()).isTrue();
    }

}