package com.myapp.videotools;

import org.assertj.core.util.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.List;


public abstract class AbstractVideoFileParser implements IVideoFileParser {

    protected final Logger log = LoggerFactory.getLogger(getClass());
    
    private File file = null;
    private VideoFile videoFile = null;
    
    protected AbstractVideoFileParser() { }
    
    @Override
    public synchronized void parse() throws IOException {
        check(videoFile);

        String videoFilePath = file.getPath();
        log.debug("      parsing metadata for: '{}' ...",videoFilePath);
        List<String> ffmpegOutputLines = readFfmpegOutput();
        initMetaDataFromLines(ffmpegOutputLines);
        log.debug("      OK, metadata for: '{}' parsed.", file.getName()); 
    }

    @Override
    public void setVideoFile(VideoFile vidfile) {
        check(vidfile);

        this.videoFile = vidfile;
        this.file = videoFile.getFile();
    }

    private void check(VideoFile vidfile) {
        if (vidfile == null)
            throw new NullPointerException();
        if (vidfile.getFile() == null || ! vidfile.getFile().isFile())
            throw new RuntimeException("cannot live without an existing file: "+vidfile);
    }


    protected abstract List<String> readVideoMetaData(File videoFile2) throws IOException;
    
    protected abstract boolean isLineOfInterest(String lineOfMetaData);
    
    protected abstract long parseDurationMillis(String lineOfMetaData);
    protected abstract long parseVideoStartOffsetMillis(String lineOfMetaData);
    protected abstract int parseTotalBytesPerSecond(String lineOfMetaData);

    protected abstract String parseVideoCodec(String lineOfMetaData);
    protected abstract String parseVideoFormat(String lineOfMetaData);
    protected abstract int parseVideoHeight(String lineOfMetaData);
    protected abstract int parseVideoWidth(String lineOfMetaData);
    protected abstract int parseVideoBytesPerSecond(String lineOfMetaData);
    protected abstract double parseVideoFrameRate(String lineOfMetaData);

    protected abstract String parseAudioCodec(String lineOfMetaData);
    protected abstract int parseAudioSampleRate(String lineOfMetaData);
    protected abstract String parseAudioChannels(String lineOfMetaData);
    protected abstract int parseAudioBytesPerSecond(String lineOfMetaData);
    
    
    
    private List<String> readFfmpegOutput() throws IOException {
        log.trace("        will now read std-out from ffmpeg-process...");
        List<String> outputLines = readVideoMetaData(file);
        
        log.trace("        OK, std-out from process read.");
        return outputLines;
    }    

    @VisibleForTesting
    protected void initMetaDataFromLines(List<String> lines) {
        log.debug("        will now parse collected output from ffmpeg...");

        lines.stream()
                .filter(this::isLineOfInterest)
                .forEach(line -> {
            gatherGeneralMetaData(line);
            gatherVideoMetaData(line);
            gatherAudioMetaData(line);
        });

        log.debug("        OK, collected output parsed. ({} lines)", lines.size());
    }  

    private void gatherGeneralMetaData(String line) {
        long durationMillis = parseDurationMillis(line);
        long offsetMillis = parseVideoStartOffsetMillis(line);
        int totalBytesPerSecond = parseTotalBytesPerSecond(line);
        
        if (durationMillis >= 0 && videoFile.getLengthMillis() < 0) {
            videoFile.setLengthMillis(durationMillis);
            log.debug("          duration       = {} ms", durationMillis);
        }
        if (offsetMillis >= 0 && videoFile.getVideoStartOffsetMillis() == 0) {
            videoFile.setVideoStartOffsetMillis(offsetMillis);            
            log.trace("          offsetMillis   = {} ms", offsetMillis);
        }
        if (totalBytesPerSecond >= 0 && videoFile.getTotalBytesPerSecond() < 0) {
            videoFile.setTotalBytesPerSecond(totalBytesPerSecond);
            log.trace("          totalBitrate   = {} b/s", offsetMillis);
        }
    }
    
    private void gatherVideoMetaData(String line) {
        String videoCodec            =  parseVideoCodec         (line);
        String videoFormat           =  parseVideoFormat        (line);
        int    videoHeight           =  parseVideoHeight        (line);
        int    videoWidth            =  parseVideoWidth         (line);
        int    videoBytesPerSecond   =  parseVideoBytesPerSecond(line);
        double videoFrameRate        =  parseVideoFrameRate     (line);

        if (videoCodec != null && videoFile.getVideoCodec() == null) {
            videoFile.setVideoCodec(videoCodec);
            log.trace("          vidCodec       = {} ", videoCodec);
        }
        if (videoFormat != null && videoFile.getVideoFormat() == null) {
            videoFile.setVideoFormat(videoFormat);
            log.trace("          vidFormat      = {} ", videoFormat);
        }
        if (videoHeight >= 0 && videoFile.getVideoHeight() < 0) {
            videoFile.setVideoHeight(videoHeight);
            log.debug("          vidHeight      = {} pixel", videoHeight);
        }
        if (videoWidth >= 0 && videoFile.getVideoWidth() < 0) {
            videoFile.setVideoWidth(videoWidth);
            log.debug("          vidWidth       = {} pixel", videoWidth);
        }
        if (videoBytesPerSecond >= 0 && videoFile.getVideoBytesPerSecond() < 0) {
            videoFile.setVideoBytesPerSecond(videoBytesPerSecond);
            log.trace("          vidBytesPerSec = {} b/s", videoBytesPerSecond);
        }
        if (videoFrameRate >= 0 && videoFile.getVideoFramesPerSecond() < 0) {
            videoFile.setVideoFramesPerSecond(videoFrameRate);
            log.trace("          vidFrameRate   = {} fps", videoFrameRate);
        }
    }
    
    private void gatherAudioMetaData(String line) {
        String  audioCodec          =  parseAudioCodec         (line);
        int     audioSampleRate     =  parseAudioSampleRate    (line);
        String  audioChannels       =  parseAudioChannels      (line);
        int     audioBytesPerSecond =  parseAudioBytesPerSecond(line);
        
        if (audioCodec != null && videoFile.getAudioCodec() == null) {
            videoFile.setAudioCodec(audioCodec);
            log.trace("          audioCodec     = {}", audioCodec);
        }
        if (audioSampleRate >= 0 && videoFile.getAudioSampleRate() < 0) {
            videoFile.setAudioSampleRate(audioSampleRate);
            log.trace("          audioSampleRate= {} Hz", audioSampleRate);
        }
        if (audioChannels != null && videoFile.getAudioChannelType() == null) {
            videoFile.setAudioChannelType(audioChannels);
            log.trace("          audioChannels  = {}", audioChannels);
        }
        if (audioBytesPerSecond >= 0 && videoFile.getAudioBytesPerSecond() < 0) {
            videoFile.setAudioBytesPerSecond(audioBytesPerSecond);
            log.trace("          audioBitrate   = {} b/s", audioBytesPerSecond);
        }
    }
}
