package com.myapp.kodi.impl.kodi_MyVideos99;

import com.myapp.kodi.common.domain.Movie;
import com.myapp.kodi.common.domain.Tag;
import com.myapp.kodi.common.domain.TvShow;
import com.myapp.kodi.common.service.KodiService;
import com.myapp.kodi.common.util.IFileWrapper;
import com.myapp.kodi.impl.kodi_MyVideos99.entities.*;
import com.myapp.kodi.impl.kodi_MyVideos99.repo.MovieRepository;
import com.myapp.kodi.impl.kodi_MyVideos99.repo.TagLinkRepository;
import com.myapp.kodi.impl.kodi_MyVideos99.repo.TagRepository;
import com.myapp.kodi.impl.kodi_MyVideos99.repo.TvShowRepository;
import com.myapp.kodi.common.service.DomainObjectMapper;
import org.springframework.beans.factory.annotation.Autowired;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.transaction.Transactional;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static com.myapp.kodi.impl.kodi_MyVideos99.repo.TagLinkRepository.MEDIA_TYPE_MOVIE;
import static java.util.stream.Collectors.toList;
import static java.util.stream.StreamSupport.stream;
import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * Implementation of the kodi backend service.
 *
 * Created by andre on 3/25/17.
 */
@Transactional
public class KodiServiceImpl implements KodiService {

    @PersistenceContext
    private EntityManager em;

    private final MovieRepository movieRepository;
    private final TagRepository tagRepository;
    private final TvShowRepository tvShowRepository;
    private final TagLinkRepository tagLinkRepository;

    @Autowired
    private DomainObjectMapper mapper;

    @Autowired
    public KodiServiceImpl(MovieRepository movieRepo, TagRepository tagRepo,
                           TvShowRepository tvShowRepo, TagLinkRepository tagLinkRepo,
                           EntityManager em) {
        this.movieRepository = movieRepo;
        this.tagRepository = tagRepo;
        this.tvShowRepository = tvShowRepo;
        this.tagLinkRepository = tagLinkRepo;
        this.em = em;
    }

    @Override
    public TvShow findByTvShowName(String name) {
        TvShowEntity hits = tvShowRepository.findByTvShowName(name);
        return mapper.convert(hits);
    }

    @Override
    public List<Movie> loadAllMovies() {
        Iterable<MovieEntity> all = movieRepository.findAll();
        return stream(all.spliterator(), false)
                .map(mapper::convert)
                .collect(Collectors.toList());
    }

    @Override
    public List<TvShow> loadAllTvShows() {
        Iterable<TvShowEntity> all = tvShowRepository.findAll();
        return stream(all.spliterator(), false)
                .map(mapper::convert)
                .collect(Collectors.toList());
    }

    @Override
    public List<Tag> loadAllTags() {
        return stream(tagRepository.findAll().spliterator(), false)
                .map(mapper::convert)
                .collect(Collectors.toList());
    }


    @Override
    public void tag(TvShow s, String tag) {
        tag(tag, TagLinkRepository.MEDIA_TYPE_TVSHOW, s.getId());
    }

    @Override
    public void tagMovie(Movie m, String tag) {
        tag(tag, MEDIA_TYPE_MOVIE, m.getId());
    }

    @Transactional
    public void tag(String tagName, String mediaType, int mediaId) {
        TagEntity tagEntity = getOrCreateTagEntity(tagName);

        List<TagLinkEntity> existing = tagLinkRepository.findAll((root, query, cb) -> cb.and(
                cb.equal(root.<Integer>get("tagId"), tagEntity.getTagId()),
                cb.equal(root.<String>get("mediaType"), mediaType),
                cb.equal(root.<Integer>get("mediaId"), mediaId)));

        switch (existing.size()) {
            case 0:
                saveTagLink(mediaType, mediaId, tagEntity.getTagId());
                return;
            case 1:
                return; // nothing to do
            default:
                throw new IllegalStateException(existing.size() + " found for " + tagName);
        }
    }

    @Override
    @Transactional
    public void tagMovies(Map<String, Set<Movie>> tagDefinitions) {
        Map<String, TagEntity> existingTags = stream(tagRepository.findAll().spliterator(), false)
                .collect(Collectors.toMap(TagEntity::getName, t -> t));

        tagDefinitions.forEach((tagName, movieSet) -> {
            TagEntity tagEntity = existingTags.computeIfAbsent(tagName, this::createTagEntity);
            movieSet.stream().filter(m -> !m.hasTag(tagName)).forEach(
                    movie -> saveTagLink(MEDIA_TYPE_MOVIE, movie.getId(), tagEntity.getTagId()));
        });
    }

    @Transactional
    public void saveTagLink(String mediaType, int mediaId, int tagId) {
        TagLinkEntity tle = new TagLinkEntity();
        tle.setMediaId(mediaId);
        tle.setMediaType(mediaType);
        tle.setTagId(tagId);
        tagLinkRepository.save(tle);
    }

    @Transactional
    public TagEntity getOrCreateTagEntity(String tagName) {
        if (isBlank(tagName)) {
            throw new IllegalArgumentException("needs a name: " + tagName);
        }

        List<TagEntity> existing = tagRepository.findAll(
                (tagEntity, query, cb) -> cb.equal(tagEntity.<String>get("name"), tagName));

        switch (existing.size()) {
            case 0: return createTagEntity(tagName);
            case 1: return existing.get(0);
            default: throw new IllegalStateException(existing.size() + " found for " + tagName);
        }
    }

    @Transactional
    public TagEntity createTagEntity(String tagName) {
        TagEntity newTag = new TagEntity();
        newTag.setName(tagName);
        return tagRepository.save(newTag);
    }

    public List<IFileWrapper> getMovieShares() {
        return getShares("movies");
    }

    public List<IFileWrapper> getTvShowShares() {
        return getShares("tvshows");
    }

    private List<IFileWrapper> getShares(String c) {
        return em.createQuery(
                "select p from PathEntity p " +
                        "where p.parent is null and p.strContent = '" + c + "'", PathEntity.class)
                .getResultList()
                .stream()
                .map(pathEntity -> {
                    return mapper.convert(pathEntity);
                })
                .collect(toList());
    }

}
