package com.myapp.kodi.common.util.smb;

import com.myapp.kodi.common.util.IFileWrapper;
import com.myapp.util.file.RecursiveTreeIterator.TreeNode;
import jcifs.smb.NtlmPasswordAuthentication;
import jcifs.smb.SmbException;
import jcifs.smb.SmbFile;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toList;

/**
 * utilizes smbFiles
 *
 * Created by andre on 8/4/15.
 */
public class SmbFileWrapper implements IFileWrapper {

    private SmbFile smbFile;
    private transient String urlAsString = null;
    private NtlmPasswordAuthentication authentication;

    public SmbFileWrapper(String url, NtlmPasswordAuthentication authentication) {
        this.authentication = authentication;
        try {
            this.smbFile = new SmbFile(url, authentication);
        } catch (MalformedURLException e) {
            throw new RuntimeException(url + " - " + authentication, e);
        }
    }

    public SmbFileWrapper(SmbFile smbFile) {
        this.smbFile = smbFile;
    }

    public SmbFileWrapper(SmbFileWrapper smbFileWrapper) {
        this.authentication = smbFileWrapper.authentication;
        this.smbFile = new SmbFile(smbFileWrapper.getURL(), authentication);
    }

    @Override
    public String getName() {
        return smbFile.getName();
    }

    @Override
    public String getPath() {
        return smbFile.getPath();
    }

    @Override
    public String getParent() {
        return smbFile.getParent();
    }

    public SmbFileWrapper getParentWrapped() {
        return new SmbFileWrapper(smbFile.getParent(), authentication);
    }

    public boolean exists() {
        try {
            return smbFile.exists();
        } catch (SmbException e) {
            throw new RuntimeException(smbFile.getName(), e);
        }
    }

    public boolean canRead() {
        try {
            return smbFile.canRead();
        } catch (SmbException e) {
            throw new RuntimeException(smbFile.getName(), e);
        }
    }

    public boolean canWrite() {
        try {
            return smbFile.canWrite();
        } catch (SmbException e) {
            throw new RuntimeException(smbFile.getName(), e);
        }
    }

    @Override
    public boolean isDirectory() {
        try {
            return smbFile.isDirectory();
        } catch (SmbException e) {
            throw new RuntimeException(smbFile.getName(), e);
        }
    }

    @Override
    public boolean isFile() {
        try {
            return smbFile.isFile();
        } catch (SmbException e) {
            throw new RuntimeException(smbFile.getName(), e);
        }
    }

    public boolean isHidden() {
        try {
            return smbFile.isHidden();
        } catch (SmbException e) {
            throw new RuntimeException(smbFile.getName(), e);
        }
    }

    public List<String> list() {
        try {
            if (!smbFile.isDirectory()) {
                return null;
            }
            return Arrays.asList(smbFile.list());
        } catch (SmbException e) {
            throw new RuntimeException(smbFile.getName(), e);
        }
    }

    @Override
    public List<IFileWrapper> listFiles() {
        try {
            if (!smbFile.isDirectory()) {
                return null;
            }
            return Stream.of(smbFile.listFiles()).map(SmbFileWrapper::new).collect(toList());
        } catch (SmbException e) {
            throw new RuntimeException(smbFile.getPath(), e);
        }
    }

    @Override
    public long getLength() {
        try {
            return smbFile.length();
        } catch (SmbException e) {
            throw new RuntimeException(smbFile.getName(), e);
        }
    }

    @Override
    public InputStream getInputStream() throws IOException {
        return smbFile.getInputStream();
    }

    public long getCreateTime() {
        try {
            return smbFile.createTime();
        } catch (SmbException e) {
            throw new RuntimeException(smbFile.getName(), e);
        }
    }

    public long getLastModifiedTime() {
        try {
            return smbFile.lastModified();
        } catch (SmbException e) {
            throw new RuntimeException(smbFile.getName(), e);
        }
    }

    @Override
    public URL getURL() {
        return smbFile.getURL();
    }

    @Override
    public String getURLAsString() {
        if (urlAsString == null) {
            urlAsString = getURL().toString();
        }
        return urlAsString;
    }

    @Override
    public final String toString() {
        return getURL().toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;

        if (!(o instanceof SmbFileWrapper)) return false;

        SmbFileWrapper that = (SmbFileWrapper) o;

        return new EqualsBuilder()
                .append(smbFile, that.smbFile)
                .isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder(17, 37)
                .append(smbFile)
                .toHashCode();
    }

    NtlmPasswordAuthentication getAuthentication() {
        return authentication;
    }

    @Override
    public Stream<IFileWrapper> streamChildrenRecursively() {
        return new SmbTreeIterator(new SmbFileNode(this)).stream().map(TreeNode::getValue);
    }
}
