package com.myapp.kodi.common.util.sftp;

import com.jcraft.jsch.*;
import com.jcraft.jsch.ChannelSftp.LsEntry;
import com.myapp.kodi.common.util.IFileWrapper;
import com.myapp.util.file.RecursiveTreeIterator;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.log4j.Logger;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Vector;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class SftpFileWrapper implements IFileWrapper {

    private static final Logger logger = Logger.getLogger(SftpFileWrapper.class);

    private SftpConnectionManager sftpConnectionManager;
    private String url;

    private transient URL cachedUrl = null;
    private List<LsEntry> cachedEntries = null;

    public SftpFileWrapper(SftpConnectionManager sftpConnectionManager, String url) {
        this.sftpConnectionManager = sftpConnectionManager;
        this.url = url;
    }

    private SftpFileWrapper(LsEntry entry, URL theUrl) {
        this.sftpConnectionManager = null;
        this.cachedEntries = Collections.singletonList(entry);
        String urlString = theUrl.toString();
        if (!urlString.endsWith("/")) {
            urlString += "/";
        }
        this.url = urlString + entry.getFilename();
    }

    SftpConnectionManager getSftpConnectionManager() {
        return sftpConnectionManager;
    }

    @Override
    public String getName() {
        String urlPath = getPath();
        int lastSlash = urlPath.lastIndexOf('/');
        return urlPath.substring(lastSlash + 1);
    }

    @Override
    public String getPath() {
        URL theUrl = getURL();
        String urlPath = theUrl.getPath();
        if (urlPath.endsWith("/")) {
            urlPath = urlPath.substring(0, urlPath.length() - 1);
        }
        if (urlPath.startsWith("//")) {
            urlPath = urlPath.substring(2);
        }
        return urlPath;
    }

    @Override
    public String getParent() {
        String urlPath = getPath();
        int lastSlash = urlPath.lastIndexOf('/');
        return urlPath.substring(0, lastSlash);
    }

    private void ls() {
        synchronized (this) {
            if (cachedEntries != null) {
                return;
            }
            ChannelSftp sftp = sftpConnectionManager.aquireSftpChannel(url);
            String path = getPath();
            try {
                Vector<?> lsResult = sftp.ls(path);

                cachedEntries = lsResult.stream()
                        .map(e -> (LsEntry) e)
                        .sorted()
                        .collect(Collectors.toList());

            } catch (SftpException e) {
                throw new RuntimeException("path: " + path, e);
            } finally {
                sftpConnectionManager.passivateSftpChannel(url, sftp);
            }
        }
    }

    @Override
    public boolean isDirectory() {
        ls();

        Optional<LsEntry> dot = cachedEntries.stream()
                .filter(e -> e.getFilename().equals("."))
                .findAny();

        return dot.isPresent() && dot.get().getAttrs().isDir();
    }

    @Override
    public boolean isFile() {
        ls();

        Optional<LsEntry> dot = cachedEntries.stream()
                .filter(e -> e.getFilename().equals("."))
                .findAny();

        LsEntry single = null;
        if (dot.isPresent()) {
            single = dot.get();
        } else if (cachedEntries.size() == 1) {
            single = cachedEntries.get(0);
        }

        return single != null && single.getAttrs().isReg();
    }

    @Override
    public String toString() {
        return url;
    }

    @Override
    public List<IFileWrapper> listFiles() {
        ls();

        List<IFileWrapper> result = cachedEntries.stream()
                .map(lse -> new SftpFileWrapper(lse, getURL()))
                .collect(Collectors.toList());

        return result;
    }

    @Override
    public long getLength() {
        if (isFile()) {
            return cachedEntries.get(0).getAttrs().getSize();
        }
        return 0;
    }

    @Override
    public InputStream getInputStream() throws IOException {
        return null; // TODO: open the sftp input stream of the file and return it
    }

    @Override
    public URL getURL() {
        if (cachedUrl == null) {
            try {
                cachedUrl = new URL(this.url);
                return cachedUrl;
            } catch (MalformedURLException e) {
                throw new RuntimeException(e);
            }
        }
        return cachedUrl;
    }

    @Override
    public String getURLAsString() {
        return url;
    }

    @Override
    public IFileWrapper getParentWrapped() {
        ls();

        if (isFile() || isDirectory()) {
            String path = getURL().toString();
            String name = getName();
            String parentPath = path.replaceAll("/" + Pattern.quote(name) + "/?$", "");
            return new SftpFileWrapper(sftpConnectionManager, parentPath);
        }

        return null;
    }

    @Override
    public Stream<IFileWrapper> streamChildrenRecursively() {
        if (isFile()) {
            return Stream.of(this);
        }
        return new SftpTreeIterator(new SftpFileNode(this)).stream()
                .map(RecursiveTreeIterator.TreeNode::getValue);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;

        if (!(o instanceof SftpFileWrapper)) return false;

        SftpFileWrapper that = (SftpFileWrapper) o;

        return new EqualsBuilder()
                .append(this.getURL(), that.getURL())
                .isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder(17, 37)
                .append(this.getURL())
                .toHashCode();
    }


    public static void main(String[] args) throws Exception {

        // create a ssh session

        SftpConnectionManager sftpConnectionManager = new SftpConnectionManager();



        File knownHostsFile = new File(new File(new File(
                System.getProperty("user.home")), ".ssh"), "known_hosts");
        sftpConnectionManager.setKnownHostsFile(knownHostsFile);
        Session session = null;

        try {
            JSch ssh = new JSch();

            session = ssh.getSession("kodiftp", "nairobi", 22);
            session.setPassword("seDiP93cF");

            SftpFileWrapper sftpDir = new SftpFileWrapper(sftpConnectionManager,
                    "sftp://kodiftp:seDiP93cF@nairobi:22//_series/ENGLISH/bobs_burgers/season_03/");
            List<IFileWrapper> files = sftpDir.listFiles();
            files.forEach(f -> System.out.println(f.getURL()));


            SftpFileWrapper sftpFile = new SftpFileWrapper(sftpConnectionManager,
                    "sftp://kodiftp:seDiP93cF@nairobi:22//_series/ENGLISH/bobs_burgers/season_03/S03E09 - God Rest Ye Merry Gentle-Ma.mp4");
            List<IFileWrapper> file = sftpFile.listFiles();
            file.forEach(f -> System.out.println(f.getURL()));


        } finally {
            if (session != null) {
                session.disconnect();
            }
        }
    }
}
