package com.myapp.kodi.common.util.sftp;

import com.jcraft.jsch.ChannelSftp;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;

import java.io.ByteArrayInputStream;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static java.util.regex.Pattern.compile;

class SftpConnectionPool {

    private static final Pattern SFTP_URL_PATTERN = compile("(?ix) ^ sftp:// ([^:]+) : ([^@]+) @ ([^:]+) (:[0-9]+) // .* $");
    private static final Logger logger = Logger.getLogger(SftpConnectionPool.class);

    private final SftpConnectionManager util;

    private String username;
    private String password;
    private String hostname;
    private int port = 22;

    private Session sshSession = null;
    private Set<ChannelSftp> activeChannels = null;
    private Set<ChannelSftp> passiveChannels = null;

    public SftpConnectionPool(String url, SftpConnectionManager util) {
        // e.g.: sftp://kodiftp:seDiP93cF@nairobi:22//_series/ENGLISH/columbo/Season 5/
        Matcher m = SFTP_URL_PATTERN.matcher(url);
        if (!m.find()) {
            throw new RuntimeException(url);
        }

        username = m.group(1);
        password = m.group(2);
        hostname = m.group(3);

        String portString = m.group(4);
        if (StringUtils.isNotBlank(portString)) {
            portString = portString.substring(1); // cut off the leading ":"
            port = Integer.parseInt(portString);
        }

        this.util = util;
    }

    public static String getKeyFromUrl(String url) {
        Matcher m = SFTP_URL_PATTERN.matcher(url);
        if (!m.find()) {
            throw new RuntimeException(url);
        }
        String username = m.group(1);
        String hostname = m.group(3);
        String portString = m.group(4);
        if (StringUtils.isBlank(portString)) {
            portString = ":22";
        }
        return username + "@" + hostname + portString;
    }

    public String getKey() {
        return username + "@" + hostname + ":" + port;
    }

    @Override
    public String toString() {
        return "Pool[" + getKey() + "]@" + hashCode();
    }


    public ChannelSftp aquireChannel() {
        if (!isConnected()) {

            // create a new ssh session

            JSch ssh = new JSch();
            try {
                byte[] knownHosts = util.readKnownHosts();
                ssh.setKnownHosts(new ByteArrayInputStream(knownHosts));
                sshSession = ssh.getSession(username, hostname, port);
                sshSession.setPassword(password);
                sshSession.connect();
                if (logger.isInfoEnabled()) {
                    logger.info("Session established to " + getKey());
                }
            } catch (Exception e) {
                throw new RuntimeException(getKey(), e);
            }
        }

        ChannelSftp channel;
        if (CollectionUtils.isNotEmpty(passiveChannels)) {
            if (logger.isDebugEnabled()) {
                logger.debug("Reusing sftp channel to " + getKey());
            }
            Iterator<ChannelSftp> iterator = passiveChannels.iterator();
            channel = iterator.next();
            iterator.remove();
        } else {
            try {
                channel = (ChannelSftp) sshSession.openChannel("sftp");
                if (logger.isDebugEnabled()) {
                    logger.debug("Create new sftp channel to " + getKey());
                }
                channel.connect();

            } catch (JSchException e) {
                throw new RuntimeException(getKey(), e);
            }
        }

        if (activeChannels == null) {
            activeChannels = new HashSet<>();
        }
        activeChannels.add(channel);
        return channel;
    }

    public boolean isConnected() {
        return sshSession != null && sshSession.isConnected();
    }

    public void passivateChannel(ChannelSftp channel) {
        activeChannels.remove(channel);
        if (passiveChannels == null) {
            passiveChannels = new HashSet<>();
        }
        passiveChannels.add(channel);
    }

    public void destroy() {
        int active = activeChannels == null ? 0 : activeChannels.size();
        int passive = passiveChannels == null ? 0 : passiveChannels.size();

        logger.info("Disconnecting ssh session to " + getKey() + " - closing " + active + " active " +
                "and " + passive + " passive channels.");

        if (active > 0) {
            logger.warn("Closing " + active + " active sessions to " + getKey() + " that " +
                    "have not been passivated.");
        }

        if (active > 0) {
            activeChannels.forEach(ChannelSftp::disconnect);
        }
        if (passive > 0) {
            passiveChannels.forEach(ChannelSftp::disconnect);
        }

        if (isConnected()) {
            sshSession.disconnect();
        }
    }
}
