package com.myapp.kodi.common.util.sftp;

import com.jcraft.jsch.ChannelSftp;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;
import org.springframework.stereotype.Service;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;
import java.util.LinkedHashMap;
import java.util.Map;

@Service
public class SftpConnectionManager {


    private static final Logger logger = Logger.getLogger(SftpFileWrapper.class);
    private static boolean sftpUrlStreamHandlerRegistered = false;
    private byte[] cachedKnownHostsBytes = null;
    private URL knownHostsResource = null;
    private final Map<String, SftpConnectionPool> pools = new LinkedHashMap<>();


    public SftpConnectionManager() {
        registerSftpUrlStreamHandler();
    }

    public ChannelSftp aquireSftpChannel(String url) {
        logger.info("Aquiring sftp channel for " + url);

        String key = SftpConnectionPool.getKeyFromUrl(url);

        synchronized (pools) {
            SftpConnectionPool existingPool = pools.get(key);
            if (existingPool == null) {
                existingPool = new SftpConnectionPool(url, this);
                pools.put(key, existingPool);
            }
            return existingPool.aquireChannel();
        }
    }

    public void passivateSftpChannel(String url, ChannelSftp channelSftp) {
        logger.info("Passivating sftp channel used for " + url);

        synchronized (pools) {
            String key = SftpConnectionPool.getKeyFromUrl(url);
            SftpConnectionPool pool = pools.get(key);
            pool.passivateChannel(channelSftp);
        }
    }

    byte[] readKnownHosts() {
        if (cachedKnownHostsBytes != null) {
            return new byte[0];
        }
        synchronized (this) {
            if (cachedKnownHostsBytes != null) {
                return new byte[0];
            }

            URL url;
            if (knownHostsResource == null) {
                url = SftpConnectionManager.class.getClassLoader().getResource("known_hosts");
            } else {
                url = knownHostsResource;
            }

            if (url == null) {
                throw new RuntimeException("known hosts file is not specified");
            }

            ByteArrayOutputStream array = new ByteArrayOutputStream();
            try {
                IOUtils.copy(url.openStream(), array);
            } catch (Exception e) {
                throw new RuntimeException("error reading " + url, e);
            }

            logger.info("Known hosts file read from resource: " + url);
            cachedKnownHostsBytes = array.toByteArray();
            return cachedKnownHostsBytes;
        }
    }

    public void setKnownHostsResource(URL knownHostsResource) {
        this.knownHostsResource = knownHostsResource;
    }

    public void setKnownHostsFile(File resource) {
        try {
            this.knownHostsResource = resource.toURI().toURL();
        } catch (MalformedURLException e) {
            throw new RuntimeException("Invalid resource: " + resource, e);
        }
    }

    public void closeAllConnections() {
        pools.values().forEach(SftpConnectionPool::destroy);
    }

    /**
     * register a dummy url stream handler factor to support sftp:// urls
     */
    public static void registerSftpUrlStreamHandler() {
        if (sftpUrlStreamHandlerRegistered) {
            return;
        }
        synchronized (SftpFileWrapper.class) {
            if (sftpUrlStreamHandlerRegistered) {
                return;
            }
            URL.setURLStreamHandlerFactory(protocol -> {
                if ("sftp".equals(protocol)) {
                    return new URLStreamHandler() {
                        @Override
                        protected URLConnection openConnection(URL u) {
                            return new URLConnection(u) {
                                @Override
                                public void connect() {
                                }
                            };
                        }
                    };
                }
                return null;
            });
            sftpUrlStreamHandlerRegistered = true;
        }
    }
}
