package com.myapp.kodi.common.util;

import com.myapp.kodi.common.domain.Movie;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.Predicate;

import static java.util.stream.Collectors.toList;

/**
 * Created by andre on 9/9/15.
 */
public class PathBasedRule implements Predicate<Movie> {

    public enum Rule {
        requireAllOf,
        requireAnyOf,
        mustNotContainAnyOf,
        mustNotContainAllOf
    }

    private List<Pair<Rule, List<String>>> rules = new ArrayList<>();

    public PathBasedRule() {
    }

    public PathBasedRule add(Rule rule, String... array) {
        if (array.length <= 0) {
            throw new IllegalArgumentException("empty search term list");
        }
        rules.add(new ImmutablePair<>(rule, Arrays.asList(array)));
        return this;
    }

    @Override
    public boolean test(Movie movie) {
        List<IFileWrapper> smbFiles = movie.getSmbFiles();
        List<String> paths = smbFiles.stream().map(IFileWrapper::getPath).collect(toList());

        for (Pair<Rule, List<String>> ruleSet : rules) {
            List<String> searchList = ruleSet.getValue();
            Rule rule = ruleSet.getKey();

            if (!testRule(rule, searchList, paths)) {
                return false;
            }
        }

        return true;
    }

    private static boolean testRule(Rule rule, List<String> searchTerms, List<String> input) {
        switch (rule) {

            case mustNotContainAllOf: { // mustNotContainAllOf
                boolean allFound = true;
                for (String search : searchTerms) {
                    if (!isFound(input, search)) {
                        allFound = false;
                        break;
                    }
                }
                if (allFound) {
                    return false;
                }
                break;
            }

            case requireAnyOf: { // requireAnyOf
                boolean anyFound = false;
                for (String search : searchTerms) {
                    if (isFound(input, search)) {
                        anyFound = true;
                        break;
                    }
                }
                if (!anyFound) {
                    return false;
                }
                break;
            }

            case mustNotContainAnyOf: { // mustNotContainAnyOf
                for (String search : searchTerms) {
                    if (isFound(input, search)) {
                        return false;
                    }
                }
                break;
            }

            case requireAllOf: { // requireAllOf
                for (String search : searchTerms) {
                    if (!isFound(input, search)) {
                        return false;
                    }
                }
                break;
            }
        }

        return true;
    }

    private static boolean isFound(List<String> list, String search) {
        boolean found = false;
        for (String path : list) {
            if (path.contains(search)) {
                found = true;
                break;
            }
        }
        return found;
    }
}
