package com.myapp.kodi.common.service;

import com.myapp.kodi.common.domain.Video;
import com.myapp.kodi.common.util.IFileWrapper;
import com.myapp.kodi.common.util.smb.SmbToLocalFileMapper;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;

import static java.time.Duration.between;
import static java.time.LocalDateTime.now;
import static java.util.stream.Collectors.toList;
import static org.springframework.util.StringUtils.getFilenameExtension;
import static org.springframework.util.StringUtils.stripFilenameExtension;

/**
 * responsible for parsing subtitles
 *
 * Created by andre on 3/20/17.
 */
@Component
public class SubtitleService {

    private static final Logger logger = Logger.getLogger(SubtitleService.class);

    private final SmbToLocalFileMapper fileMapper;

    @Autowired
    public SubtitleService(SmbToLocalFileMapper fileMapper) {
        this.fileMapper = fileMapper;
    }


    public boolean hasLocalSubtitle(IFileWrapper file) {
        String filePath = file.getPath();
        String filePathWithoutExt = stripFilenameExtension(filePath);

        if (filePathWithoutExt.equals(filePath)) {
            logger.warn("filename without extension!? " + filePath);
            return false;
        }
        if (hasEquallyNamedSubtitleFile(file)) {
            logger.debug("found equally named subtitle for:   " + filePath);
            return true;
        }
        if (hasEmbeddedSubtitleStream(file)) {
            return true;
        }

        if (logger.isTraceEnabled()) {
            logger.trace("No subtitles found for " + filePath);
        }
        return false;
    }

    private boolean hasEquallyNamedSubtitleFile(IFileWrapper file) {
        String filePath = file.getPath();
        String filePathWithoutExt = stripFilenameExtension(filePath);
        List<IFileWrapper> siblings = file.getParentWrapped().listFiles();

        return siblings.stream().anyMatch(s -> {
            String sPath = s.getPath();
            if (sPath.equalsIgnoreCase(filePath)) {
                return false; // same file
            }

            if (!sPath.toLowerCase().contains(filePathWithoutExt.toLowerCase())) {
                return false;
            }

            String ext = getFilenameExtension(sPath);
            if (ext == null) {
                logger.warn("filename without extension!? " + sPath);
                return false;
            }

            switch (ext.toLowerCase()) {
                case "sub":
                case "srt":
                case "sbv":
                case "lrc":
                case "mpsub":
                    return true;
                default:
                    return false;
            }
        });
    }

    /**
     * IMPLEMENTATION NOTE<br/>
     * <ul>
     * <li>we are only interested in english subtitles.</li>
     * <li>we are only interested in subs for english videos or multilingual movies
     * (as they always include an english audio stream too, in my collection)</li>
     * <li>so, if a movie is english or multilingual and it contains embedded subs,
     * we assume those subs are english too.</li>
     * </ul>
     * this way, we don't have to parse "english" from the subtitle stream description,
     * we can return true as we found any subtitle stream in the video file.
     **/
    private boolean hasEmbeddedSubtitleStream(IFileWrapper file) {
        File fsFile = fileMapper.toLocalFile(file);
        BufferedReader r = null;
        try {
            Process avconv = new ProcessBuilder()
                    .directory(fsFile.getParentFile())
                    .command("avconv", "-i", fsFile.getName())
                    .redirectErrorStream(true)
                    .start();
            avconv.waitFor();

            r = new BufferedReader(new InputStreamReader(avconv.getInputStream()));
            for (String line; (line = r.readLine()) != null; ) {
                if (logger.isTraceEnabled()) {
                    logger.trace(line);
                }
                line = line.toLowerCase();


                if (line.contains("stream") && line.contains("subtitle")
                        && line.matches(".*\\ben(|g|lisc?h)\\b.*")) {
                    logger.debug("found embedded subtitles in: " + file.getName() + " - " + line);
                    return true;
                }
            }

        } catch (Exception e) {
            throw new RuntimeException(fsFile.getAbsolutePath(), e);
        } finally {
            if (r != null) {
                try {
                    r.close();
                } catch (IOException ignored) {
                }
            }
        }
        return false;
    }


    public <V extends Video> List<V> getEnglishVideosWithoutSubs(Collection<V> englishVideos) {
        LocalDateTime start = now();
        AtomicInteger checked = new AtomicInteger();
        AtomicInteger haveSubs = new AtomicInteger();
        AtomicInteger noSubs = new AtomicInteger();

        List<V> withoutSubtitles = englishVideos.parallelStream().filter((e) -> {
            Optional<IFileWrapper> optional = e.getFirstSmbFile();
            if (!optional.isPresent()) {
                logger.warn("every FileConnected instance should have a file: " + e);
                return false;
            }

            IFileWrapper file = optional.get();
            boolean hasNoSubtitle = !hasLocalSubtitle(file);
            if (hasNoSubtitle) {
                noSubs.incrementAndGet();
            } else {
                haveSubs.incrementAndGet();
            }
            if (checked.incrementAndGet() % 100 == 0) {
                logStats(between(start, now()).toMillis(), englishVideos.size(), haveSubs, noSubs, checked);
            }
            return hasNoSubtitle;
        }).collect(toList());

        logStats(between(start, now()).toMillis(), englishVideos.size(), haveSubs, noSubs, checked);
        return withoutSubtitles;
    }

    private void logStats(long durationMillis, int elementCount,
                          Number have, Number haveNot, Number processed) {
        long milliPerElement = durationMillis / processed.intValue();
        logger.info("Filtered " + elementCount + " elements in " + durationMillis + " ms. "
                + processed + " were processed, " + have + " elements had subs, "
                + haveNot + " didn't. speed " + milliPerElement + " ms/element");
    }
}
