package com.myapp.kodi.common.service;

import com.myapp.kodi.common.domain.Episode;
import com.myapp.kodi.common.domain.Movie;
import com.myapp.kodi.common.domain.Tag;
import com.myapp.kodi.common.domain.TvShow;
import com.myapp.kodi.common.util.IFileWrapper;
import com.myapp.kodi.common.util.sftp.SftpFileWrapper;
import com.myapp.kodi.common.util.sftp.SftpConnectionManager;
import com.myapp.kodi.common.util.smb.SmbFileWrapper;
import com.myapp.kodi.impl.kodi_MyVideos99.entities.*;
import jcifs.smb.NtlmPasswordAuthentication;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.myapp.kodi.common.service.KodiService.UNPARSEABLE_INT_VALUE;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.startsWithIgnoreCase;

/**
 * util that converts kodi entites into domain objects
 * <p>
 * Created by andre on 3/25/17.
 */
@Component
public final class DomainObjectMapper {

    @Autowired(required = false) // only needed in case we access a sftp file system
    private SftpConnectionManager sftpConnectionManager;

    public Movie convert(MovieEntity me) {
        Integer year = parseIntegerIfPossible(me.getReleaseYear());
        Integer runtimeSecInt = parseIntegerIfPossible(me.getRuntimeSeconds());
        List<IFileWrapper> smbFiles = convert(me.getFile());
        List<Tag> tags = me.getTags().stream().map(this::convert).collect(toList());

        Movie movie = new Movie(me.getIdMovie());
        movie.setTitle(me.getTitle());
        movie.setPlot(me.getPlot());
        movie.setReleaseYear(year == null ? UNPARSEABLE_INT_VALUE : year);
        movie.setImdbId(me.getImdbId());
        movie.setSmbFiles(smbFiles);
        movie.setRuntimeSeconds(runtimeSecInt == null ? UNPARSEABLE_INT_VALUE : runtimeSecInt);
        movie.setOriginalTitle(me.getOriginalTitle());
        movie.setTags(tags);
        return movie;
    }

    private Episode convert(EpisodeEntity ee) {
        Integer season = parseIntegerIfPossible(ee.getSeason());
        Integer episodeNr = parseIntegerIfPossible(ee.getEpisode());
        List<IFileWrapper> smbFiles = convert(ee.getFile());

        Episode episode = new Episode(ee.getIdEpisode());
        episode.setTitle(ee.getTitle());
        episode.setSeason(season == null ? UNPARSEABLE_INT_VALUE : season);
        episode.setEpisodeNr(episodeNr == null ? UNPARSEABLE_INT_VALUE : episodeNr);
        episode.setSmbFiles(smbFiles);
        episode.setPlot(ee.getPlot());
        return episode;
    }


    public TvShow convert(TvShowEntity tve) {
        Integer year = parseIntegerIfPossible(tve.getFirstAired());

        TvShow show = new TvShow(tve.getIdShow());
        show.setTitle(tve.getTitle());
        show.setPlot(tve.getPlot());
        show.setFirsAired(year == null ? UNPARSEABLE_INT_VALUE : year);
        tve.getEpisodes().stream().map(this::convert).forEach(show::addEpisode);
        return show;
    }


    public IFileWrapper convert(PathEntity pathEntity) {
        String strPath = StringUtils.trimToEmpty(pathEntity.getStrPath());
        if (!startsWithIgnoreCase(strPath, "smb://") && !startsWithIgnoreCase(strPath, "sftp://")) {
            return null;
        }
        return createFileWrapper(strPath);
    }


    private static final Pattern STACKED_PATHS_PATTERN =
            Pattern.compile("(?ix) ( (?:smb|sftp):// .*? ) \\s* (?=( $ | , \\s* (?:smb|sftp):// ))");


    private List<IFileWrapper> convert(FileEntity fileEntity) {
        if (fileEntity == null) {
            return Collections.emptyList();
        }

        String strFilename = fileEntity.getStrFilename();
        if (StringUtils.isBlank(strFilename)) {
            return Collections.emptyList();
        }

        if (startsWithIgnoreCase(strFilename, "stack://")) {
            Matcher m = STACKED_PATHS_PATTERN.matcher(strFilename);
            if (!m.find()) {
                throw new RuntimeException("stack but not containing paths: " + strFilename);
            }

            List<IFileWrapper> list = new LinkedList<>();
            do {
                String smbPath = m.group(1);
                list.add(createFileWrapper(smbPath));
            } while (m.find());
            return Collections.unmodifiableList(list);
        }

        String strPath = fileEntity.getPath().getStrPath();
        if (!strPath.endsWith("/")) {
            strPath += "/";
        }
        strPath += strFilename;

        IFileWrapper singleFile = createFileWrapper(strPath);
        return Collections.singletonList(singleFile);
    }

    public Tag convert(TagEntity tag) {
        Tag t = new Tag(tag.getTagId());
        t.setName(tag.getName());
        return t;
    }

    private static Integer parseIntegerIfPossible(String intString) {
        return StringUtils.isNumeric(intString) ? Integer.parseInt(intString) : null;
    }

    public IFileWrapper createFileWrapper(String url) {
        if (url.startsWith("smb://")) {
            return new SmbFileWrapper(url, NtlmPasswordAuthentication.ANONYMOUS);
        }
        if (url.startsWith("sftp://")) {
            return new SftpFileWrapper(sftpConnectionManager, url);
        }
        throw new UnsupportedOperationException("unknown file type: '" + url + "'");
    }

}
