package com.myapp.kodi;

import com.myapp.kodi.common.domain.Movie;
import com.myapp.kodi.common.domain.Video;
import com.myapp.kodi.common.util.IFileWrapper;
import org.apache.commons.lang3.StringUtils;
import org.jboss.logging.Logger;

import java.util.*;

import static java.util.Comparator.comparing;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.trimToEmpty;

/**
 * various string operations
 *
 * Created by andre on 4/2/17.
 */
class PrintUtil {

    private static final String LINE_SEPARATOR = System.getProperty("line.separator");
    private static final Logger logger = Logger.getLogger(PrintUtil.class);


    static class Diff {
        Set<IFileWrapper> dbFilesUniq;
        List<IFileWrapper> dbFilesFlat;
        Set<IFileWrapper> fsFilesUniq;
        Set<IFileWrapper> inDbButNotInFs = new LinkedHashSet<>();
        Set<IFileWrapper> inFsButNotInDb = new LinkedHashSet<>();

    }

    static <T extends Video> Diff calcDifferences(Map<T, List<IFileWrapper>> filesPerItem,
                                                  List<IFileWrapper> inFs) {
        Diff diff = new Diff();

        diff.dbFilesFlat = filesPerItem.values().stream().flatMap(List::stream).collect(toList());
        diff.dbFilesUniq = new HashSet<>(diff.dbFilesFlat);
        diff.fsFilesUniq = new HashSet<>(inFs);

        Set<IFileWrapper> inDbButNotInFs = diff.inDbButNotInFs;
        inDbButNotInFs.addAll(diff.dbFilesUniq);
        inDbButNotInFs.removeAll(diff.fsFilesUniq);

        Set<IFileWrapper> inFsButNotInDb = diff.inFsButNotInDb;
        inFsButNotInDb.addAll(diff.fsFilesUniq);
        inFsButNotInDb.removeAll(diff.dbFilesUniq);

        return diff;
    }

    static <T extends Video> Diff showDifferences(String type,
                                                  Map<T, List<IFileWrapper>> filesPerItem,
                                                  List<IFileWrapper> inFs) {
        final String prefix = type.toUpperCase() + ": ";
        Diff diff = calcDifferences(filesPerItem, inFs);

        logger.info(prefix + "Show differences of files in database and in filesystem.");

        logger.info(prefix + "Files registered in database: " + diff.dbFilesUniq.size() +
                ", files found in filesystem: " + diff.fsFilesUniq.size());

        Map<IFileWrapper, Set<Video>> itemsPerFile = new LinkedHashMap<>();
        filesPerItem.forEach((mediaItem, fileList) -> fileList.forEach(file -> {
            itemsPerFile.computeIfAbsent(file, x -> new LinkedHashSet<>()).add(mediaItem);
        }));


        if (diff.dbFilesUniq.size() != diff.dbFilesFlat.size()) {
            logger.warn(prefix + "FILES IN DATABASE MIGHT CONTAIN DUPLICATES:");
            Map<IFileWrapper, List<IFileWrapper>> duplicatesMap = new LinkedHashMap<>();

            for (IFileWrapper f : diff.dbFilesUniq) {
                if (duplicatesMap.containsKey(f)) {
                    continue;
                }
                List<IFileWrapper> duplicates = diff.dbFilesUniq.stream()
                        .filter(f::equals)
                        .sorted(comparing(sfw -> sfw == null ? "" : trimToEmpty(sfw.getPath())))
                        .collect(toList());
                assert duplicates.size() > 0;
                if (duplicates.size() > 1) {
                    duplicatesMap.put(f, duplicates);
                }
            }

            String sep = LINE_SEPARATOR + "  ";
            duplicatesMap.forEach((file, duplicateList) -> {
                Set<Video> mediaItems = itemsPerFile.getOrDefault(file, Collections.emptySet());
                if (mediaItems.size() == duplicateList.size()) {
                    // only print a debug message, this can happen in a video database
                    // e.g. Monk S01E01-02 - Mr. Monk and the Candidate Part 1 and 2.avi
                    if (logger.isDebugEnabled()) {
                        logger.debug("File \"" + file.getName() + "\" mapped to multiple videos: " +
                                mediaItems.stream().map(Object::toString).collect(joining(", ")));
                    }
                } else {
                    logger.warn(duplicateList.stream().map(IFileWrapper::getName)
                            .collect(joining(sep, "Duplicates: " + sep, "")));
                }
            });
        }

        if (diff.fsFilesUniq.size() != inFs.size()) {
            logger.warn(prefix + "FILES IN FILESYSTEM MIGHT CONTAIN DUPLICATES.");
            for (int i = 0; i < inFs.size(); i++) {
                IFileWrapper f = inFs.get(i);
                if (inFs.lastIndexOf(f) != i) {
                    logger.warn("Duplicate in filesystem: " + f.getPath());
                }
            }
        }

        if (diff.inDbButNotInFs.size() > 0) {
            logger.info(prefix + diff.inDbButNotInFs.size() + " FILES WERE NOT FOUND IN FILESYSTEM:");
            diff.inDbButNotInFs.forEach(f -> logger.warn("Referenced file not found: " + f));
        }

        if (diff.inFsButNotInDb.size() > 0) {
            logger.warn(prefix + diff.inFsButNotInDb.size() + " FILES WERE NOT FUND IN DATABASE:");
            diff.inFsButNotInDb.forEach(f -> logger.warn("File is not in database: " + f));
        }

        return diff;
    }

    static void displaySearchResults(List<String> searchExpressions,
                                     Map<String, List<Movie>> foundMovies) {
        StringBuilder b = new StringBuilder();

        int longestExprLen = searchExpressions.stream().mapToInt(String::length).max()
                .orElseThrow(() -> new IllegalArgumentException("empty? " + searchExpressions));

        for (String wt : searchExpressions) {
            List<Movie> hits = foundMovies.get(wt);
            b.setLength(0);

            if (hits == null) {
                b.append("    ");
            } else {
                assert hits.size() > 0;
                b.append(":-) ");
            }
            b.append("Expression ");
            b.append(StringUtils.rightPad("'" + wt + "'", longestExprLen + 2));

            if (hits == null) {
                b.append(" did not match any movie. Searched for: ").append(App.toWords(wt));
            } else {
                assert hits.size() > 0;
                b.append(" matched ").append(hits.size()).append(" movies. ").append(hits);
            }

            logger.info(b);
        }
    }
}
