package com.myapp.kodi;

import com.myapp.kodi.common.domain.Movie;
import com.myapp.kodi.common.domain.TvShow;
import com.myapp.kodi.common.domain.Video;
import com.myapp.kodi.common.service.KodiService;
import com.myapp.kodi.common.util.IFileWrapper;
import com.myapp.kodi.common.util.sftp.SftpConnectionManager;
import com.myapp.kodi.common.util.TaggingStrategy;
import org.apache.commons.cli.*;
import org.jboss.logging.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import static java.util.regex.Pattern.compile;
import static java.util.stream.Collectors.*;
import static org.apache.commons.lang3.StringUtils.trimToEmpty;

/**
 * Hello world!
 */
@Component
public class App {

    private static final Logger logger = Logger.getLogger(App.class);

    private final KodiService kodi;
    private final TaggingStrategy taggingStrategy;

    private List<Movie> allMoviesCached;
    private List<TvShow> allTvShowsCached;
    public static final Pattern BLACKLISTED_EXTS = compile("(?ix).* \\. (jpe?g|idx|sub|ac3|srt|nfo|txt|sfv|ssa|log|sh) $");
    public static final Pattern BLACKLISTED_DIRS = compile("(?ix).* /EXTRAS/ .*");

    @Autowired(required = false) // only needed when sftp is used
    private SftpConnectionManager sftpConnectionManager;

    private App(KodiService kodi, TaggingStrategy taggingsy) {
        this.kodi = kodi;
        this.taggingStrategy = taggingsy;
    }

    public static void main(String[] args) throws Throwable {
        App app = createApplication();

        int result;
        long startTime = System.currentTimeMillis();
        try {
            result = invokeBusinessLogic(args, app);
        } finally {
            if (app.sftpConnectionManager != null) {
                app.sftpConnectionManager.closeAllConnections();
            }
        }
        long finishTime = System.currentTimeMillis();
        long executionTime = finishTime - startTime;
        logger.debug("executionTime = " + executionTime);
        System.exit(result);
    }

    public static int invokeBusinessLogic(String[] args, App app) throws ParseException {
        Options opts = new Options();
        opts.addOption("ls", "list-roots", false, "List media file roots");
        opts.addOption("shows", "display-show-status", false, "List shows that are unclean");
        opts.addOption("movies", "display-movie-status", false, "List movies that are unclean");
        opts.addOption("tagmovies", "display-movie-status", false, "Creates movie tags");
        opts.addOption("moviesearch", "search-movies", true, "Search through movies");
        opts.addOption("testmoviesearch", "search-movies", false, "Test the search through movies");

        CommandLineParser parser = new DefaultParser();
        CommandLine cmd = parser.parse(opts, args);
        if (cmd.hasOption("ls")) {
            app.printMediaFileRoots();
            return 0;
        }

        if (cmd.hasOption("shows")) {
            app.invokeShowBusinessLogic();
            return 0;
        }

        if (cmd.hasOption("movies")) {
            app.invokeMovieBusinessLogic();
            return 0;
        }

        if (cmd.hasOption("tagmovies")) {
            app.createMovieTags();
            return 0;
        }

        if (cmd.hasOption("moviesearch")) {
            String searchString = cmd.getOptionValue("moviesearch");
            List<String> pattern = Collections.singletonList(searchString);
            Map<String, List<Movie>> result = app.searchVariousMovies(pattern);
            System.out.println("Found " + result.size() + " matches for " + searchString + ":");
            result.forEach((searchStr, list) -> {
                list.forEach(m -> {
                    System.out.println(m.getTitle() + " - " + m.getSmbFilesToString());
                });
            });
            return 0;
        }

        if (cmd.hasOption("testmoviesearch")) {
            app.searchVariousMovies();
            return 0;
        }

        new HelpFormatter().printHelp("Kodi-Helper", opts);
        return 1;
    }

    @SuppressWarnings("WeakerAccess")
    public static App createApplication() {
        logger.debug("init spring context...");
        AbstractApplicationContext c =
                new ClassPathXmlApplicationContext("applicationContext.xml");
        logger.debug("spring context initialized.");


        logger.debug("create application...");
        App app = new App(
                c.getBean(KodiService.class),
                c.getBean(TaggingStrategy.class)
        );
        c.getAutowireCapableBeanFactory().autowireBean(app);
        logger.debug("application created.");

        return app;
    }

    /**
     * execute movie related business logic
     */
    private void invokeShowBusinessLogic() {

        logger.debug("Fetching all episodes' files from database...");
        LinkedHashMap<Video, List<IFileWrapper>> db = getAllTvShowFilesInDatabaseAsMap();

        logger.debug("Fetching all episodes' files from filesystem...");
        List<IFileWrapper> inFileSystem = getAllTvShowVideoFilesInFileSystem();

        PrintUtil.showDifferences("episodes", db, inFileSystem);
    }

    /**
     * execute movie related business logic
     */
    private void invokeMovieBusinessLogic() {
        logger.debug("fetching all movies' files from database...");
        Map<Video, List<IFileWrapper>> inDb = getAllMovieFilesInDatabaseAsMap();

        logger.debug("fetching all movies' files from filesystem...");
        List<IFileWrapper> inFs = getAllMovieVideoFilesInFileSystem();
        PrintUtil.showDifferences("movies", inDb, inFs);
    }

    /**
     * tags movies as they are specified by the taggingStrategy
     */
    @SuppressWarnings("WeakerAccess")
    public void createMovieTags() {
        logger.debug("Creating all movie tags...");
        Map<String, Predicate<Movie>> tagRules = taggingStrategy.getMovieTagRules();
        List<Movie> allMovies = loadAllMoviesCached();

        Map<String, Set<Movie>> moviesByTagName = new LinkedHashMap<>();
        for (Movie m : allMovies) {
            tagRules.forEach((tagName, predicate) -> {
                if (predicate.test(m)) {
                    moviesByTagName.computeIfAbsent(tagName, foo -> new LinkedHashSet<>()).add(m);
                }
            });
        }

        logger.info("Linking " +
                moviesByTagName.values().stream().flatMap(Collection::stream).distinct().count() +
                " movies to " + moviesByTagName.size() + " tags. TagsLinks to be created: " +
                moviesByTagName.values().stream().mapToInt(Set::size).sum());

        kodi.tagMovies(moviesByTagName);
    }

    private List<Movie> loadAllMoviesCached() {
        if (allMoviesCached == null) {
            allMoviesCached = kodi.loadAllMovies();
        }
        return allMoviesCached;
    }

    private List<TvShow> loadAllTvShowsCached() {
        if (allTvShowsCached == null) {
            allTvShowsCached = kodi.loadAllTvShows();
        }
        return allTvShowsCached;
    }

    public TvShow findByTvShowName(String name) {
        return kodi.findByTvShowName(name);
    }

    /**
     * prints the media library shares.
     */
    @SuppressWarnings("WeakerAccess")
    public void printMediaFileRoots() {
        List<IFileWrapper> roots = kodi.getAllShares();
        logger.info("MediaFile Roots: (" + roots.size() + ")");
        roots.forEach(path -> logger.info("shared directory found: " + path));
    }

    /**
     * search in movies for a hardcoded set of search expressions
     */
    private void searchVariousMovies() {
        List<String> searchExpressions = Arrays.asList(
                "Batman vs superman 2016",
                "What if (2014)",
                "In bruges",
                "Tim and erics million dollar movie",
                "13 assassins",
                "Mad max",
                "Fargo",
                "Nightcrawler",
                "Aloys",
                "Green room",
                "Blue ruin",
                "Swiss army man",
                "Lost in translation",
                "Dr strange",
                "History of Violence. ",
                "Mothman",
                "The Secret in Their Eyes",
                "Sicario",
                "Casino",
                "Trash",
                "District 8",
                "Miracle_at_St._Anna",
                "Conspiracy",
                "Suicide squad",
                "Bad lieutnant",
                "Gruber geht",
                "Was hat uns bloss so ruiniert",
                "Hotel rock n roll",
                "Bösterreich",
                "Schlawiner",
                "Hinterholz 8",
                "Braunschlag",
                "The Necessary Death of Charlie Countryman"
        );
        Map<String, List<Movie>> foundMovies = searchVariousMovies(searchExpressions);
        PrintUtil.displaySearchResults(searchExpressions, foundMovies);
    }

    @SuppressWarnings("WeakerAccess")
    public Map<String, List<Movie>> searchVariousMovies(List<String> searchExpressions) {
        if (new HashSet<>(searchExpressions).size() != searchExpressions.size()) {
            throw new RuntimeException("Duplicate search expressions: " + searchExpressions);
        }
        logger.info("Searching for " + searchExpressions.size() + " search terms in movies...");

        Map<String, List<String>> wantedWords = searchExpressions.stream().collect(
                toMap(o -> o, App::toWords, (a, b) -> a, LinkedHashMap::new));
        Map<String, List<Movie>> result = new LinkedHashMap<>();

        loadAllMoviesCached().stream().parallel().forEach(m -> {
            List<String> movieWords = toWords(
                    trimToEmpty(m.getTitle()) + " " + trimToEmpty(m.getOriginalTitle()));

            wantedWords.forEach((String term, List<String> words) -> {
                Set<String> matches = words.stream().filter(movieWords::contains).collect(toSet());
                if (matches.size() == words.size()) {
                    logger.debug(m + " matched search term " + term);
                    List<Movie> list = result.get(term);
                    if (list == null) {
                        synchronized (result) {
                            list = result.computeIfAbsent(term, foo -> new ArrayList<>());
                        }
                    }
                    list.add(m);
                }
            });
        });

        logger.debug("Search ended. Hits: " + result.size() + ", movies found: " +
                result.values().stream().flatMap(Collection::stream).distinct().count());

        return result;
    }

    /**
     * converts a search term to a list of words
     *
     * @param term the search term
     * @return the list of words that can be compared for a match.
     */
    static List<String> toWords(String term) {
        return Stream.of(term
                .toLowerCase()
                .replaceAll("(19|20)[0-9][0-9]", " ")
                .replaceAll("[^a-z0-9]", " ")
                .replaceAll("\\b(and|a|st|vs|the|in|of|der|das|at)\\b", " ")
                .replaceAll("\\s+", " ")
                .trim()
                .split("\\s+")
        ).distinct().sorted().collect(toList());
    }



    public List<IFileWrapper> getAllTvShowVideoFilesInFileSystem() {
        return getAllVideoFilesInFileSystem(kodi.getTvShowShares());
    }

    private LinkedHashMap<Video, List<IFileWrapper>> getAllTvShowFilesInDatabaseAsMap() {
        List<Video> tvShows = loadAllTvShowsCached().stream()
                .flatMap(s -> s.getEpisodes().stream())
                .map(e -> (Video) e)
                .collect(toList());
        return getVideoFilesAsMap(tvShows);
    }

    public Stream<IFileWrapper> filterBlackList(Stream<IFileWrapper> inputStream) {
        return inputStream
                .filter(f -> {
                    String urlAsString = f.getURLAsString();
                    return !BLACKLISTED_EXTS.matcher(urlAsString).matches()
                            && !BLACKLISTED_DIRS.matcher(urlAsString).matches();
                });
    }

    public List<IFileWrapper> getFiles(IFileWrapper parentDir) {
        Stream<IFileWrapper> blackListFiltered = filterBlackList(
                parentDir.streamChildrenRecursively()
                        .filter(IFileWrapper::isFile)
        );
        return blackListFiltered.sorted().collect(toList());
    }

    private List<IFileWrapper> getAllVideoFilesInFileSystem(List<IFileWrapper> roots) {
        Stream<IFileWrapper> blackListFiltered = filterBlackList(
                roots.parallelStream()
                        .flatMap(IFileWrapper::streamChildrenRecursively)
                        .filter(IFileWrapper::isFile)
        );
        return blackListFiltered.sorted().collect(toList());
    }

    static LinkedHashMap<Video, List<IFileWrapper>> getVideoFilesAsMap(List<Video> videos) {
        return videos.stream().collect(toMap((Video m) -> m, Video::getSmbFiles, (a, b) -> {
            Set<IFileWrapper> uniq = new LinkedHashSet<>(a.size() + b.size());
            uniq.addAll(a);
            uniq.addAll(b);
            return new ArrayList<>(uniq);
        }, LinkedHashMap::new));
    }


    public List<IFileWrapper> getAllMovieVideoFilesInFileSystem() {
        return getAllVideoFilesInFileSystem(kodi.getMovieShares());
    }

    private LinkedHashMap<Video, List<IFileWrapper>> getAllMovieFilesInDatabaseAsMap() {
        List<Video> movies = loadAllMoviesCached().stream().map(e -> (Video) e).collect(toList());
        return getVideoFilesAsMap(movies);
    }

    public SftpConnectionManager getSftpConnectionManager() {
        return sftpConnectionManager;
    }

    public void setSftpConnectionManager(SftpConnectionManager sftpConnectionManager) {
        this.sftpConnectionManager = sftpConnectionManager;
    }
}
